package io.transwarp.cluster.comparison;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import io.transwarp.common.bean.node.NodeBean;
import io.transwarp.common.bean.service.ConfigBean;
import io.transwarp.common.bean.service.RoleBean;
import io.transwarp.common.bean.service.ServiceBean;
import io.transwarp.common.util.UtilTools;

public class CalcExecutorMemoryAndCore {
	
	private final Map<String, Double> resourceCores;
	private final ServiceBean service;
	private final Map<String, StringBuffer> executorResources;
	
	private String[] workerHosts;
	private Map<String, NodeBean> workerNodes;
	

	public CalcExecutorMemoryAndCore(final Map<String, Double> resourceCores,
			final ServiceBean service) {
		this.resourceCores = resourceCores;
		this.service = service;
		this.executorResources = new HashMap<String, StringBuffer>();
		this.workerHosts = new String[]{};
		this.workerNodes = new HashMap<String, NodeBean>();
	}
	
	public void calc() {
		try {
			Map<String, String> fileValues = getServiceConfig("ngmr-context-env.sh");
			if(fileValues == null) {
				return;
			}
			getWorkers(fileValues);
			calcMemoryAndCore(fileValues);
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			addExecutorRoleToService();
		}
	}
	
	private Map<String, String> getServiceConfig(final String filename) throws Exception {
		ConfigBean config = service.getConfig();
		String serverNode = null;
		List<RoleBean> roles = service.getRoles();
		List<String> nodes = new ArrayList<String>();
		for (RoleBean role : roles) {
			String roleType = role.getRoleType();
			NodeBean node = role.getNode();
			if (roleType.equals("INCEPTOR_SERVER")) {
				serverNode = node.getIpAddress();
			} else if(roleType.indexOf("WORKER") != -1) {
				nodes.add(node.getHostName());
				workerNodes.put(node.getHostName(), node);
			}
		}
		if (serverNode == null) {
			return null;
		}
		int number = nodes.size();
		this.workerHosts = new String[number];
		this.workerHosts = nodes.toArray(workerHosts);
		return config.getConfigFile(serverNode, filename);
	}
	
	private void getWorkers(final Map<String, String> fileValues) throws Exception {
		String number_executors = fileValues.get("INCEPTOR_YARN_NUMBER_EXECUTORS");
		String number_workers = fileValues.get("INCEPTOR_YARN_NUMBER_WORKERS");
		String executor_list = fileValues.get("NGMR_YARN_EXECUTOR_LIST");
		if (number_workers.equals("-1")) {
			if (executor_list == null || executor_list.equals("")) {
				if (number_executors.equals("-1")) {  //每个计算节点一个executor
					addExecutorNumberToDatanodes(workerHosts, "1");
				} else {  //每个计算节点executor数：executor总数/计算节点数
					int nodeNumber = workerHosts.length;
					addExecutorNumberToDatanodes(workerHosts, number_executors + "/" + nodeNumber);
				}
			} else {
				String[] hostnames = executor_list.split(",");
				if (number_executors.equals("-1")) {  //指定节点上有1个executor
					addExecutorNumberToDatanodes(hostnames, "1");
				} else {  //指定节点上executor数：executor总数/指定节点数
					int nodeNumber = hostnames.length;
					addExecutorNumberToDatanodes(hostnames, number_executors + "/" + nodeNumber);
				}
			}
		} else {
			if (number_executors.equals("-1")) {  //每个计算节点executor数：worker数/计算节点数
				int nodeNumber = workerHosts.length;
				addExecutorNumberToDatanodes(workerHosts, number_workers + "/" + nodeNumber);
			} else {  //每个计算节点executor数：executor总数/worker总数
				addExecutorNumberToDatanodes(workerHosts, number_executors + "/" + number_workers);
			}
		}
	}
	
	private void addExecutorNumberToDatanodes(String[] hosts, String number) {
		for (String host : hosts) {
			StringBuffer result = new StringBuffer();
//			result.append(",active number : ").append(service.getActiveExecutorByHost(host));
			result.append(",allocated number : ").append(number);
			this.executorResources.put(host, result);			
		}
	}
	
	private void calcMemoryAndCore(final Map<String, String> fileValue) throws Exception {
		String scheduler_mode = fileValue.get("INCEPTOR_YARN_SCHEDULER_MODE");
		if (scheduler_mode.indexOf("Heterogeneous") != -1) {
			String mem_cpu_ratio = fileValue.get("INCEPTOR_YARN_EXECUTOR_MEM_CPU_RATIO");
			String used_resource_percent = fileValue.get("INCEPTOR_YARN_EXECUTOR_USED_RESOURCE_PERCENT");
			calcMemoryAndCoreInRatioMode(mem_cpu_ratio, used_resource_percent);
		} else {
			String executor_cores = fileValue.get("INCEPTOR_YARN_EXECUTOR_CORES");
			String executor_memory = fileValue.get("INCEPTOR_YARN_EXECUTOR_MEMORY");
			calcMemoryAndCoreInFixedMode(executor_cores, executor_memory);
		}		
	}
	
	private void calcMemoryAndCoreInRatioMode(String mem_cpu_ratio, String used_resource_percent) throws Exception {
		Double mem_cpu = Double.valueOf(mem_cpu_ratio);
		Double used_percent = Double.valueOf(used_resource_percent);
		for (Entry<String, StringBuffer> entry : executorResources.entrySet()) {
			String hostname = entry.getKey();
			StringBuffer buffer = entry.getValue();
			Double totalCores = this.resourceCores.get(hostname);
			Double usedCores = totalCores * used_percent;
			Double usedMemory = usedCores * mem_cpu * 1024;
			buffer.append(",core : ").append(usedCores.intValue()).append(",Memory : ").append(UtilTools.getCarrySize(usedMemory.longValue()));
		}
	}
	
	private void calcMemoryAndCoreInFixedMode(String executor_cores, String executor_memory) throws Exception {
		Double cores = Double.valueOf(executor_cores);
		Double memory = Double.valueOf(UtilTools.numberFormat(executor_memory)) * 1024 * 1024;
		for (Entry<String, StringBuffer> entry : executorResources.entrySet()) {
			StringBuffer buffer = entry.getValue();
			buffer.append(",core : ").append(cores).append(",Memory : ").append(UtilTools.getCarrySize(memory.longValue()));
		}
	}
	
	private void addExecutorRoleToService() {
		for (Entry<String, StringBuffer> entry : executorResources.entrySet()) {
			String hostname = entry.getKey();
			StringBuffer buffer = entry.getValue();
			RoleBean role = new RoleBean();
			role.setRoleType("INCEPTOR_EXECUTOR");
			role.setName(String.format("executor (%s)", hostname));
			role.setNode(this.workerNodes.get(hostname));
			role.setResource(buffer.toString());
			service.addRole(role);
		}
	}
}
