package io.transwarp.cluster.comparison;

import io.transwarp.common.GlobalArgs;
import io.transwarp.common.bean.InspectionData;
import io.transwarp.common.bean.node.NodeBean;
import io.transwarp.common.bean.service.RoleBean;
import io.transwarp.common.bean.service.ServiceBean;
import io.transwarp.common.util.UtilTools;
import io.transwarp.report.comparison.Comparison;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import net.sf.json.JSONObject;

public class RoleMapComparisonImpl implements Comparison {

	private final InspectionData data;
	private final ComparisonResultOfCluster comparisonResult;
	
	private Map<String, Double> resourceCores;
	private int count;
	
	public RoleMapComparisonImpl(final ComparisonResultOfCluster comparisonResult,
			final InspectionData data) {
		this.data = data;
		this.comparisonResult = comparisonResult;
		this.count = 1;
		this.resourceCores = new HashMap<String, Double>();
	}
	
	public void comparison() {
		Map<String, List<String>> nodeList = listNodeOfRack();
		listRoleOnNode();
		mergeSameNode(nodeList);
	}
	
	private Map<String, List<String>> listNodeOfRack() {
		Map<String, List<String>> nodeOrderOfRack = new HashMap<String, List<String>>();
		Map<String, NodeBean> nodes = data.nodeInfo.getNodes();
		for (Entry<String, NodeBean> entry : nodes.entrySet()) {
			String hostname = entry.getKey();
			NodeBean node = entry.getValue();
			String rackname = node.getRackName();
			List<String> nodeList = nodeOrderOfRack.get(rackname);
			if (nodeList == null) {
				nodeList = new ArrayList<String>();
				nodeOrderOfRack.put(rackname, nodeList);
			}
			String totalMemory = UtilTools.getCarrySize(node.getTotalPhysMemBytes());
			String totalCores = node.getCpu();
			comparisonResult.roleResourceOnNodes.put(hostname, new ResourceOnNode(totalMemory, totalCores));
			nodeList.add(hostname);
		}
		return nodeOrderOfRack;
	}
	
	private void listRoleOnNode() {
		ServiceBean[] services = sortServices(data.clusterInfo.getServices());
		comparisonResult.orderServices = services;
		for (ServiceBean service : services) {
			if (GlobalArgs.version.startsWith("4") && service.getType().equals("INCEPTOR_SQL")) {
				CalcExecutorMemoryAndCore calcExecutor = new CalcExecutorMemoryAndCore(this.resourceCores, service);
				calcExecutor.calc();
			}
			String serviceTopic = String.format("%s[%s]", service.getName(), service.getHealth());
			List<RoleBean> roles = service.getRoles();
			for (RoleBean role : roles) {
				String roleType = role.getRoleType();
				NodeBean node = role.getNode();
				String hostname = node.getHostName();
				String rackname = node.getRackName();
				String value = role.getHealth();
				Integer activeExecutor = service.getActiveExecutorByHost(hostname);
				if (roleType.equals("INCEPTOR_EXECUTOR") && activeExecutor != null) {
					value = String.format("%s,active number : %s%s", value, activeExecutor, role.getResource());
				} else {
					value += role.getResource();
				}
				addRoleResource(rackname, hostname, serviceTopic, roleType, value);
			}
		}
	}
	
	private ServiceBean[] sortServices(Map<String, ServiceBean> services) {
		int serviceNumber = services.size();
		ServiceBean[] orderServices = new ServiceBean[serviceNumber];
		int index = 0;
		for (Entry<String, ServiceBean> entry : services.entrySet()) {
			orderServices[index++] = entry.getValue();
		}
		Arrays.sort(orderServices, new Comparator<ServiceBean>() {
			@Override
			public int compare(ServiceBean service1, ServiceBean service2) {
				int id1 = Integer.valueOf(service1.getId());
				int id2 = Integer.valueOf(service2.getId());
				if (id1 > id2) {
					return 1;
				} else {
					return -1;
				}
			}
		});
		return orderServices;
	}
	
	private void addRoleResource(final String rackname, 
			final String hostname,
			final String serviceTopic,
			final String roleType,
			final String roleInfo) {
		List<String> serviceRoleList = comparisonResult.serviceRoleMap.get(serviceTopic);
		if (serviceRoleList == null) {
			serviceRoleList = new ArrayList<String>();
			comparisonResult.serviceRoleMap.put(serviceTopic, serviceRoleList);
			comparisonResult.serviceList.add(serviceTopic);
		}
		int index = getIndexOfServiceRoleList(serviceTopic, roleType);
		ResourceOnNode resourceInfo = comparisonResult.roleResourceOnNodes.get(hostname);
		resourceInfo.addNewResource(index, roleInfo);
		if (roleInfo.indexOf("resource-memory") != -1) {
			String[] items = roleInfo.split(",");
			for (String item : items) {
				if (item.startsWith("resource-cpu")) {
					this.resourceCores.put(hostname, UtilTools.getValueFromSize(item)*-1);
				}
			}
		}
	}
	
	private int getIndexOfServiceRoleList(final String servicename, final String roleType) {
		int index = 0;
		for (String name : comparisonResult.serviceList) {
			List<String> roleList = comparisonResult.serviceRoleMap.get(name);
			if (name.equals(servicename)) {
				int roleIndex = roleList.indexOf(roleType);
				if (roleIndex == -1) {
					index += roleList.size();
					roleList.add(roleType);
				} else {
					index += roleIndex;
				}
				break;
			} else {
				index += roleList.size();
			}
		}
		return index;
	}
	
	private void mergeSameNode(Map<String, List<String>> nodeLists) {
		for (Entry<String, List<String>> entry : nodeLists.entrySet()) {
			String rackname = entry.getKey();
			List<String> nodeList = entry.getValue();
			List<JSONObject> mergeResultOnRack = mergeSameNodeOnSameRack(nodeList);
			comparisonResult.mergeResult.put(rackname, mergeResultOnRack);
		}
	}
	
	private List<JSONObject> mergeSameNodeOnSameRack(final List<String> nodeList) {
		List<JSONObject> mergeResultOnRack = new ArrayList<JSONObject>();
		int nodeNumber = nodeList.size();
		boolean[] isMerge = new boolean[nodeNumber];
		for (int i = 0; i < nodeNumber; i++) {
			if (isMerge[i]) {
				continue;
			} else {
				isMerge[i] = true;
			}
			String nodeI = nodeList.get(i);
			ResourceOnNode resourceI = comparisonResult.roleResourceOnNodes.get(nodeI);
			String otherName = "node" + (count++);
			StringBuffer hosts = new StringBuffer();
			hosts.append(nodeI);
			for (int j = i + 1; j < nodeNumber; j++) {
				if (isMerge[j]) {
					continue;
				}
				String nodeJ = nodeList.get(j);
				ResourceOnNode resourceJ = comparisonResult.roleResourceOnNodes.get(nodeJ);
				if (resourceI.equals(resourceJ)) {
					isMerge[j] = true;
					hosts.append(",").append(nodeJ);
				}
			}
			JSONObject result = new JSONObject();
			result.put("otherName", otherName);
			result.put("hosts", hosts.toString());
			mergeResultOnRack.add(result);			
		}
		return mergeResultOnRack;
	}
}
