package io.transwarp.cluster.report;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import net.sf.json.JSONObject;
import io.transwarp.cluster.comparison.ComparisonResultOfCluster;
import io.transwarp.cluster.comparison.ResourceOnNode;
import io.transwarp.report.xls.ReportEnum;
import io.transwarp.report.xls.XlsReport;
import jxl.CellView;
import jxl.write.Label;
import jxl.write.WritableSheet;
import jxl.write.WritableWorkbook;

public class RoleMapXlsReportImpl implements XlsReport {

	private final WritableSheet sheet;
	private final ComparisonResultOfCluster comparisonResult;
	
	private int maxCol = 0;
	
	public RoleMapXlsReportImpl(final WritableWorkbook workbook,
			final ReportEnum checkItem,
			final ComparisonResultOfCluster comparisonResult) {
		this.sheet = workbook.createSheet("roleMap", checkItem.ordinal());
		this.comparisonResult = comparisonResult;
	}
	
	@Override
	public void writeSheet() throws Exception {
		List<String> serviceList = comparisonResult.serviceList;
		Map<String, List<String>> serviceRoleMap = comparisonResult.serviceRoleMap;
		Map<String, List<JSONObject>> nodeListOfMerge = comparisonResult.mergeResult;
		Map<String, ResourceOnNode> resourceInfos = comparisonResult.roleResourceOnNodes;
		
		int beginRow = 0;
		beginRow = writeNodeOtherNameTable(beginRow, nodeListOfMerge);
		beginRow = writeServiceRoleMap(beginRow, serviceList, serviceRoleMap, nodeListOfMerge, resourceInfos);
		setColumnSize();
	}
	
	/*
	 * 填写别名和节点hostname对应表单
	 */
	private int writeNodeOtherNameTable(int beginRow, final Map<String, List<JSONObject>> nodeListOfMerge) throws Exception {
		int column = 0;
		sheet.addCell(new Label(column++, beginRow++, "节点列表", CELL_BOLD));
		sheet.addCell(new Label(column, beginRow, "rack name", CELL_BOLD));
		sheet.addCell(new Label(column + 1, beginRow, "node other name", CELL_BOLD));
		sheet.addCell(new Label(column + 2, beginRow, "hostname", CELL_BOLD));
		beginRow += 1;

		for (Entry<String, List<JSONObject>> entry : nodeListOfMerge.entrySet()) {
			String rackname = entry.getKey();
			List<JSONObject> otherNameList = entry.getValue();
			int beginMerge = beginRow;
			for (JSONObject json : otherNameList) {
				String otherName = json.getString("otherName");
				String hosts = json.getString("hosts");
				sheet.addCell(new Label(column + 1, beginRow, otherName, CELL_THIN));
				sheet.addCell(new Label(column + 2, beginRow, hosts, CELL_THIN));
				beginRow += 1;
			}
			if (beginMerge < beginRow - 1) {
				sheet.mergeCells(column, beginMerge, column, beginRow - 1);
			}
			sheet.addCell(new Label(column, beginMerge, rackname, CELL_THIN));
		}
		this.maxCol = Math.max(maxCol, column);
		return beginRow;
	}
	
	/*
	 * 填写角色分布列表
	 */
	private int writeServiceRoleMap(int beginRow,
			final List<String> serviceList,
			final Map<String, List<String>> serviceRoleMap,
			final Map<String, List<JSONObject>> nodeListOfMerge,
			final Map<String, ResourceOnNode> resourceInfos) throws Exception {
		List<String> nodeOrder = writeTableHeader(beginRow, nodeListOfMerge);
		beginRow += 4;
		int index = 0;
		int column = 1;
		for (String servicename : serviceList) {
			int serviceBeginMerge = beginRow;
			List<String> roleList = serviceRoleMap.get(servicename);
			for (String roleType : roleList) {
				int roleBeginMerge = beginRow;
				beginRow = writeRoleResource(beginRow, column + 2, index++, nodeOrder, resourceInfos);
				if (roleBeginMerge < beginRow - 1) {
					sheet.mergeCells(column + 1, roleBeginMerge, column + 1, beginRow - 1);
				}
				sheet.addCell(new Label(column + 1, roleBeginMerge, roleType, CELL_BOLD));
			}
			if (serviceBeginMerge < beginRow - 1) {
				sheet.mergeCells(column, serviceBeginMerge, column, beginRow - 1);
			}
			sheet.addCell(new Label(column, serviceBeginMerge, servicename, CELL_BOLD));
		}
		beginRow = writeNodeTotalResource(beginRow, nodeOrder, resourceInfos);
		return beginRow;
	}
	
	/*
	 * 填写服务角色分布图的表头
	 * @return 按序排列的节点hostname
	 */
	private List<String> writeTableHeader(int beginRow, final Map<String, List<JSONObject>> nodeListOfMerge) throws Exception {
		int column = 0;
		sheet.addCell(new Label(column++, beginRow++, "服务角色列表", CELL_BOLD));
		
		sheet.mergeCells(column, beginRow, column + 1, beginRow);
		sheet.addCell(new Label(column, beginRow, "rack name", CELL_BOLD));
		
		sheet.mergeCells(column, beginRow + 1, column + 1, beginRow + 1);
		sheet.addCell(new Label(column, beginRow + 1, "node other name", CELL_BOLD));
		
		sheet.mergeCells(column, beginRow + 2, column + 1, beginRow + 2);
		sheet.addCell(new Label(column, beginRow + 2, "title", CELL_BOLD));
		
		column += 2;
		List<String> nodeOrder = new ArrayList<String>();
		for (Entry<String, List<JSONObject>> entry : nodeListOfMerge.entrySet()) {
			int beginMerge = column;
			String rackname = entry.getKey();
			List<JSONObject> nodeList = entry.getValue();
			for (JSONObject json : nodeList) {
				String otherName = json.getString("otherName");
				String hosts = json.getString("hosts");
				sheet.mergeCells(column, beginRow + 1, column + 1, beginRow + 1);
				sheet.addCell(new Label(column, beginRow + 1, otherName, CELL_THIN));
				sheet.addCell(new Label(column++, beginRow + 2, "status", CELL_THIN));
				sheet.addCell(new Label(column++, beginRow + 2, "resource", CELL_THIN));
				nodeOrder.add(hosts.split(",")[0]);
			}
			if (beginMerge < column - 1) {
				sheet.mergeCells(beginMerge, beginRow, column - 1, beginRow);
			}
			sheet.addCell(new Label(beginMerge, beginRow, rackname, CELL_THIN));
		}
		this.maxCol = Math.max(column, maxCol);
		return nodeOrder;
	}
	
	private int writeRoleResource(int beginRow,
			int beginCol, 
			final int index, 
			final List<String> nodeList, 
			final Map<String, ResourceOnNode> resourceInfos) throws Exception {
		int maxRow = beginRow;
		for (String hostname : nodeList) {
			int row = beginRow;
			ResourceOnNode resourceInfo = resourceInfos.get(hostname);
			String resource = resourceInfo.getResource(index);
			String[] items = resource.split(",");
			sheet.addCell(new Label(beginCol, row, items[0], CELL_THIN));
			int number = items.length;
			for (int i = 1; i < number; i++) {
				sheet.addCell(new Label(beginCol + 1, row++, items[i], CELL_THIN));
			}
			if (number <= 1) {
				row += 1;
			}
			beginCol += 2;
			maxRow = Math.max(maxRow, row);
		}
		return maxRow;
	}
	
	private int writeNodeTotalResource(int beginRow, List<String> nodeOrder, Map<String, ResourceOnNode> resourceInfos) throws Exception {
		sheet.mergeCells(1, beginRow, 1, beginRow + 3);
		sheet.addCell(new Label(1, beginRow, "node resource", CELL_BOLD));
		sheet.addCell(new Label(2, beginRow, "Total Physics Memory", CELL_BOLD));
		sheet.addCell(new Label(2, beginRow + 1, "Total physics Core", CELL_BOLD));
		sheet.addCell(new Label(2, beginRow + 2, "Allocated Memory", CELL_BOLD));
		sheet.addCell(new Label(2, beginRow + 3, "Allocated Core", CELL_BOLD));
		int column = 4;
		for (String hostname : nodeOrder) {
			ResourceOnNode resource = resourceInfos.get(hostname);
			sheet.addCell(new Label(column, beginRow, resource.getTotalMemory(), CELL_THIN));
			sheet.addCell(new Label(column, beginRow + 1, resource.getTotalCores(), CELL_THIN));
			sheet.addCell(new Label(column, beginRow + 2, resource.getAllocatedMemory(), CELL_THIN));
			sheet.addCell(new Label(column, beginRow + 3, resource.getAllocatedCores(), CELL_THIN));
			column += 2;
		}
		return beginRow + 4;
	}
	
	private void setColumnSize() {
		CellView cellView = new CellView();
		cellView.setAutosize(true);
		cellView.setFormat(CELL_THIN);
		for (int i = 1; i < maxCol; i++) {
			sheet.setColumnView(i, cellView);
		}
	}
}
