package io.transwarp.common.util;

import io.transwarp.common.GlobalArgs;
import io.transwarp.common.bean.service.ConfigBean;

import java.net.InetSocketAddress;
import java.net.URI;
import java.util.LinkedList;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Queue;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.hdfs.HAUtil;
import org.apache.hadoop.security.UserGroupInformation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HdfsConnUtils {

	private final static Logger log = LoggerFactory.getLogger(HdfsConnUtils.class);
	private boolean buildPoolSuccess = false;
	private boolean openKerberos = false;
	private Configuration config;
	private String activeNamenodeIP;
	private String activeNamenodeHost;
	private int port;
	private Queue<FileSystem> fsPool;
	
	public boolean openHdfsConnPool(final boolean openKerberos,
			final String managerHost,
			final ConfigBean serviceConfig,
			final String configIP,
			final String keytab,
			final int hdfsConnNumber) {
		this.openKerberos = openKerberos;
		this.fsPool = new LinkedList<FileSystem>();
		
		try {
			getConfiguration(serviceConfig, configIP, managerHost, keytab);
			getActiveNamenode();
			getHdfsConnected(hdfsConnNumber);
			this.buildPoolSuccess = true;
		} catch (Exception e) {
			this.buildPoolSuccess = false;
			String errorMessage = "open hdfs connected pool is faild | " + e.getMessage();
			GlobalArgs.ERROR_INFO.add(errorMessage);
			e.printStackTrace();
		} 			
		return this.buildPoolSuccess;
	}
	
	public boolean isBuildPoolSuccess() {
		return this.buildPoolSuccess;
	}
	
	public FileSystem getHdfsConn() throws Exception {
		boolean getConn = false;
		FileSystem fs = null;
		while (!getConn) {
			synchronized (fsPool) {
				if (fsPool.size() > 0) {
					fs = fsPool.poll();
					getConn = true;
					break;
				}
			}
			while (fs == null) {
				Thread.sleep(1000);
			}
		}
		return fs;
	}
	
	public void closeHdfsConn(FileSystem fs) {
		if (fs == null) {
			return;
		}
		synchronized (fsPool) {
			fsPool.offer(fs);
		}
	}
	
	public void closeHdfsPool() throws Exception {
		if (!buildPoolSuccess) {
			return;
		}
		while (!fsPool.isEmpty()) {
			FileSystem fs = fsPool.poll();
			fs.close();
		}
		buildPoolSuccess = false;
	}
	
	public static String getHdfsPath(String path) {
		path = path.trim();
		if (!path.startsWith("hdfs://")) {
			return path; 
		} else {
			StringBuffer buffer = new StringBuffer();
			String[] temps = path.split("/");
			for (int i = 3; i < temps.length; i++) {
				buffer.append("/").append(temps[i]);
			}
			if (buffer.length() == 0) {
				buffer.append("/");
			}
			return buffer.toString();
		}
	}
	
	public String getNamenodeIP() {
		return this.activeNamenodeIP;
	}
	
	public String getNamenodeHost() {
		return this.activeNamenodeHost;
	}
	
	private void getConfiguration(final ConfigBean serviceConfig,
			final String configIP,
			final String managerHost,
			final String keytab) throws Exception {
		config = new Configuration();
		String[] filenames = new String[]{"hdfs-site.xml", "core-site.xml"};
		for (String filename : filenames) {
			Map<String, String> fileValue = serviceConfig.getConfigFile(configIP, filename);
			for (Entry<String, String> entry : fileValue.entrySet()) {
				config.set(entry.getKey(), entry.getValue());
			}
		}
		if (openKerberos) {
			if (managerHost == null) {
				log.error("manager node hostname is null, please check manager ip");
			}
			UserGroupInformation.setConfiguration(config);
			UserGroupInformation.loginUserFromKeytab("hdfs/" + managerHost.toLowerCase(Locale.getDefault()), keytab);
		}
	}
	
	private void getActiveNamenode() throws Exception {
		FileSystem fs = FileSystem.get(config);
		InetSocketAddress active = HAUtil.getAddressOfActive(fs);
		this.activeNamenodeIP = active.getAddress().getHostAddress();
		this.activeNamenodeHost = active.getHostName().toLowerCase(Locale.getDefault());
		this.port = active.getPort();
		fs.close();
	}
	
	private void getHdfsConnected(Integer hdfsConnNumber) throws Exception {
		for (int i = 0; i < hdfsConnNumber; i++) {
			if (openKerberos) {
				fsPool.add(FileSystem.get(config));
			} else {
				fsPool.add(FileSystem.get(new URI(String.format("hdfs://%s:%s", activeNamenodeIP, port)), config, "hdfs"));
			}
		}
	}
	
}
