package io.transwarp.common.util;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.LinkedList;
import java.util.Locale;
import java.util.Queue;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.GZIPOutputStream;

import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.io.IOUtils;

public final class UtilTools {
	
	private final static String[] UNIT = new String[]{"B", "KB", "MB", "GB", "TB", "PB"};

	public static void buildDir(String path) {
		File file = new File(path);
		if (!file.exists()) {
			file.mkdirs();
		}
	}
	
	public static void deleteFileOrDir(String path) {
		File file = new File(path);
		if (!file.exists()) {
			return;
		}
		
		if (file.isFile()) {
			file.delete();
		}
		
		if (file.isDirectory()) {
			File[] children = file.listFiles();
			for (File child : children) {
				deleteFileOrDir(child.getAbsolutePath());
			}
			file.delete();
		}
	}
	
	public static String readFile(String path) throws Exception {
		StringBuffer buffer = new StringBuffer();
		FileReader fileRead = new FileReader(path);
		BufferedReader reader = new BufferedReader(fileRead);
		String line = null;
		while (true) {
			line = reader.readLine();
			if (line == null) {
				break;
			}
			buffer.append(line).append("\n");
		}
		reader.close();
		fileRead.close();
		return buffer.toString();
	}
	
	public synchronized static String numberFormat(String value) {
		StringBuffer buffer = new StringBuffer();
		Pattern pattern = Pattern.compile("[-0-9.]+");
		Matcher matcher = pattern.matcher(value);
		while (matcher.find()) {
			buffer.append(matcher.group());
		}
		if (buffer.toString().length() == 0) {
			return "0";
		}
		return buffer.toString();
	}
	
	public static Double getValueFromSize(String size) {
		String tolowString = size.toLowerCase(Locale.getDefault());
		Double value = Double.valueOf(UtilTools.numberFormat(size));
		if (tolowString.endsWith("kb")) {
			value = value * 1024;
		} else if (tolowString.endsWith("mb")) {
			value = value * 1024 * 1024;
		} else if (tolowString.endsWith("gb")) {
			value = value * 1024 * 1024 * 1024;
		} else if (tolowString.endsWith("tb")) {
			value = value * 1024 * 1024 * 1024 * 1024;
		} else if (tolowString.endsWith("pb")) {
			value = value * 1024 * 1024 * 1024 * 1024 * 1024;
		}
		return value;
	}
	
	public static String getCarrySize(Object value) {
		Double number = getValueFromSize(value.toString());
		int id = 0;
		while (number > 1024) {
			id += 1;
			number /= 1024;
		}
		return String.format("%.2f %s", number, UNIT[id]);
	}
	
	/**
	 * 将指定的文件夹压缩成tar.gz格式的压缩文件，输出路径为与要压缩文件夹所在的目录下
	 * @param filePath 要进行压缩的文件夹路径
	 * @throws Exception
	 */
	public static void compressTarGz(String filePath) throws Exception {
		// 获取压缩文件的路径和文件名称的分割点
		if (filePath.endsWith("/")) {
			filePath = filePath.substring(0, filePath.length() - 1);
		}
		SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
		String timeStamp = format.format(new Date());
		String targeName = filePath + "-" + timeStamp;
		try {
			if (!new File(targeName + ".tar.gz").exists()) {
				UtilTools.deleteFileOrDir(targeName + ".tar.gz");
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		int dirIndex = filePath.lastIndexOf("/");
		
		// 将文件夹下的所有文件打包为一个tar包
		FileOutputStream tarOutput = new FileOutputStream(targeName + ".tar");
		TarArchiveOutputStream tarArchive = new TarArchiveOutputStream(tarOutput);
		File sourceFile = new File(filePath);
		Queue<File> queue = new LinkedList<File>();
		queue.offer(sourceFile);
		while (!queue.isEmpty()) {
			File file = queue.poll();
			if (file.isDirectory()) {
				File[] children = file.listFiles();
				for (File child : children) {
					queue.add(child);
				}
			} else {
				TarArchiveEntry entry = new TarArchiveEntry(file);
				// 设置tar包中文件的路径
				String path = file.getAbsolutePath().substring(dirIndex);
				entry.setName(path);
				tarArchive.putArchiveEntry(entry);
				IOUtils.copy(new FileInputStream(file), tarArchive);
				tarArchive.closeArchiveEntry();
			}
		}
		if (tarArchive != null) {
			tarArchive.flush();
			tarArchive.close();
		}
		
		// 将tar包压缩为tar.gz
		FileInputStream input = new FileInputStream(targeName + ".tar");
		FileOutputStream goalFile = new FileOutputStream(targeName + ".tar.gz");
		GZIPOutputStream gzipOutput = new GZIPOutputStream(goalFile);
		IOUtils.copy(input, gzipOutput);
		input.close();
		gzipOutput.flush();
		gzipOutput.close();
		
		// 删除tar包
		File tarFile = new File(targeName + ".tar");
		tarFile.delete();
	}
	
	private UtilTools() {
	}
}
