package io.transwarp.hdfs.comparison;

import java.util.Map;

import io.transwarp.common.GlobalArgs;
import io.transwarp.common.util.UtilTools;
import io.transwarp.report.comparison.Comparison;

public class HdfsInfoComparisonImpl implements Comparison {
	
	private final Map<String, String> hdfsInfo;
	private final ComparisonResultOfHdfs comparisonResult;
	private double maxUsedPercent;
	private double minUsedPercent;

	public HdfsInfoComparisonImpl(final ComparisonResultOfHdfs comparisonResult,
			final Map<String, String> hdfsInfo) {
		this.comparisonResult = comparisonResult;
		this.hdfsInfo = hdfsInfo;
		maxUsedPercent = 0.0;
		minUsedPercent = Double.MAX_VALUE;
	}
	
	@Override
	public void comparison() throws Exception {
		checkReport();
		checkFsck();
	}
	
	private void checkReport() throws Exception {
		String[] lines = hdfsInfo.get("report").split("\n");
		String nodeName = "total";
		String nodeStatus = "";
		String usedSize = "";
		String usedPercent = "";
		for (String line : lines) {
			line = line.replaceAll("\t", " ").trim();
			int splitID = line.indexOf(": ");
			if (splitID == -1) {
				splitID = line.length() - 1;
			}
			if (line.length() == 0) {
				continue;
			}
			String key = line.substring(0, splitID);
			String value = line.substring(splitID + 1);
			if (key.startsWith("DFS Used%")) {
				usedPercent = value;
				savePercent(value);
				comparisonResult.report.add(new String[]{nodeName, nodeStatus, usedSize, usedPercent});
			} else if (key.startsWith("DFS Used")) {
				usedSize = value;
			} else if (key.startsWith("Hostname")) {
				nodeName = value;
			} else if (key.startsWith("Decommission Status")) {
				nodeStatus = value;
			}
		}
		Double usedPercentLimit = Double.valueOf(GlobalArgs.prop_rules.getProperty("hdfs.space.MaxUsedPercent.limit", "75"));
		Double dataSkewLimit = Double.valueOf(GlobalArgs.prop_rules.getProperty("hdfs.space.dataSkew.limit", "10"));
		if (maxUsedPercent > usedPercentLimit || maxUsedPercent - minUsedPercent > dataSkewLimit) {
			comparisonResult.reportWarn = true;
		}
	}

	private void checkFsck() {
		String[] lines = hdfsInfo.get("fsck").split("\n");
		for (String line : lines) {
			line = line.replaceAll("\t", " ").trim();
			int splitID = line.indexOf(": ");
			if (splitID == -1) {
				splitID = line.length() - 1;
			}
			if (line.length() == 0) {
				continue;
			}
			String key = line.substring(0, splitID);
			String value = line.substring(splitID + 1);
			if (key.startsWith("Total")) {
				comparisonResult.fsck.add(new String[]{key, value});
			} else if (key.startsWith("Missing replicas")) {
				comparisonResult.fsck.add(new String[]{key, value});
				String[] items = value.split(" ");
				if ("0".equals(items[0])) {
					comparisonResult.fsckWarn = true;
				}
			}
		}
	}
	
	private void savePercent(String value) {
		Double percent = Double.valueOf(UtilTools.numberFormat(value));
		maxUsedPercent = Math.max(maxUsedPercent, percent);
		minUsedPercent = Math.min(minUsedPercent, percent);
	}
}

