package io.transwarp.hdfs.servlet;

import java.util.Stack;
import java.util.concurrent.atomic.AtomicInteger;

import net.sf.json.JSONObject;

import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.transwarp.common.GlobalArgs;
import io.transwarp.common.bean.ClusterInfo;
import io.transwarp.common.util.AbstractRunnable;
import io.transwarp.common.util.HdfsConnUtils;
import io.transwarp.common.util.UtilTools;
import io.transwarp.inspection.CheckItemEnum;

public class HdfsSpaceCheckRunnable extends AbstractRunnable {
	
	private static Logger log = LoggerFactory.getLogger(HdfsSpaceCheckRunnable.class);
	private final HdfsConnUtils hdfsPool;
	private final ClusterInfo clusterInfo;
	private final String[] tableSpaces = new String[]{
			"/inceptor\\S*/user\\S*", 
			"/hyperbase\\S*/data\\S*"
	};
	private FileSystem fs;
	private CountRunnable count;
	private Long smallFileLimit;
	private Integer smallFileNumber;

	public HdfsSpaceCheckRunnable(final HdfsConnUtils hdfsPool,
			final ClusterInfo clusterInfo) {
		this.clusterInfo = clusterInfo;
		this.hdfsPool = hdfsPool;
		count = new CountRunnable();
	}

	@Override
	public void startCheck() throws Exception {
		log.info("begin check hdfs space");
		getLimitValue();
		if (!hdfsPool.isBuildPoolSuccess()) {
			throw new Exception("hdfs connection pool is no open");
		}
		fs = hdfsPool.getHdfsConn();
		count.start();
		deepSearchMenu(fs, "/");
	}

	@Override
	public void dealWithError(Exception e) {
		String errorMessage = String.format("check hdfs space faild | " + e.getMessage());
		GlobalArgs.ERROR_INFO.add(errorMessage);
		log.error(errorMessage);
		e.printStackTrace();		
	}

	@Override
	public void closeThread() {
		log.info("hdfs space check completed");
		hdfsPool.closeHdfsConn(fs);
		count.stopCount();
		GlobalArgs.timeStatisticses.get(CheckItemEnum.SERVICE.name()).setEndTime();
	}
	
	private void getLimitValue() {
		smallFileLimit = Long.valueOf(GlobalArgs.prop_rules.getProperty("cluster.smallFile.size.limit", "1"))*1024*1024;
		smallFileNumber = Integer.valueOf(GlobalArgs.prop_rules.getProperty("hdfs.smallFile.number.limit", "1000"));
	}
	
	private void deepSearchMenu(FileSystem fs, String path) throws Exception {
		long childTotalSize = 0L;
		Stack<HdfsSpaceDir> stack = new Stack<HdfsSpaceDir>();
		stack.add(new HdfsSpaceDir(path, fs.listStatus(new Path(path))));
		while (!stack.isEmpty()) {
			boolean addResult = true;
			HdfsSpaceDir dir = stack.pop();
			dir.addMenu(childTotalSize);
			childTotalSize = 0L;
			while (dir.hasNext()) {
				FileStatus file = dir.nextChild();
				if (file == null) {
					break;
				}
				String filePath = HdfsConnUtils.getHdfsPath(file.getPath().toString());
				if (isTableSpace(filePath)) {
					continue;
				}
				if (file.isDirectory()) {
					CountRunnable.completedMenuAndFile.incrementAndGet();
					HdfsSpaceDir childDir = new HdfsSpaceDir(filePath, fs.listStatus(file.getPath()));
					stack.add(dir);
					stack.add(childDir);
					addResult = false;
					break;
				} else {
					dir.addFile(file.getLen());
				}
			}
			if (!dir.hasNext() && addResult) {
				CountRunnable.completedMenuAndFile.incrementAndGet();
				int smallFile = dir.getSmallFileNumber();
				int totalNumber = dir.getTotalFileNumber();
				String itemPath = dir.getPath();
				long totalSize = dir.getTotalSize();
				if (smallFile > this.smallFileNumber || itemPath.equals("/")) {
					JSONObject json = new JSONObject();
					json.put("path", itemPath);
					json.put("totalSize", UtilTools.getCarrySize(totalSize));
					json.put("smallFileNumber", smallFile);
					json.put("totalNumber", totalNumber);
					clusterInfo.addHdfsSpace(json);
				}
				childTotalSize = dir.getTotalSize();
			}
		}
	}
	
	private boolean isTableSpace(String path) {
		for (String tableSpace : tableSpaces) {
			if (path.matches(tableSpace)) {
				return true;
			}
		}
		return false;
	}
	
	private class HdfsSpaceDir {
		private String path;
		private FileStatus[] children;
		private int totalChild;
		private int iterator;
		private int totalFiles;
		private int smallFileNumber;
		private Long totalSize;
		
		public HdfsSpaceDir(String path, FileStatus[] children) {
			this.path = path;
			this.children = children;
			this.totalChild = children.length;
			this.iterator = 0;
			this.totalFiles = 0;
			this.smallFileNumber = 0;
			this.totalSize = 0L;
			
		}
		
		public boolean hasNext() {
			if (iterator < totalChild) {
				return true;
			} else {
				return false;
			}
		}
		
		public FileStatus nextChild() {
			if (iterator < totalChild) {
				return children[iterator++];
			} else {
				return null;
			}
		}
		
		public void addFile(Long size) {
			totalFiles += 1;
			if (size < smallFileLimit) {
				smallFileNumber += 1;
			}
			totalSize += size;
		}
		
		public void addMenu(Long size) {
			this.totalSize += size;
		}
		
		public Long getTotalSize() {
			return this.totalSize;
		}
		
		public int getSmallFileNumber() {
			return this.smallFileNumber;
		}
		
		public int getTotalFileNumber() {
			return this.totalFiles;
		}
		
		public String getPath() {
			return this.path;
		}
	}
	
	private static class CountRunnable extends Thread {
		
		private static final AtomicInteger completedMenuAndFile = new AtomicInteger(0);
		private static boolean stop = false;
		
		public CountRunnable() {
		}
		
		@Override
		public void run() {
			while (!stop) {
				int value = completedMenuAndFile.intValue();
				log.info("completed check of hdfs dir and file is " + value);
				try {
					Thread.sleep(3000);
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
			int value = completedMenuAndFile.intValue();
			log.info("hdfs space is completed, total of dir and file is " + value);
		}
		
		public void stopCount() {
			stop = true;
		}
	}
}
