package io.transwarp.inspection;

import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Locale;
import java.util.Map.Entry;

import org.apache.log4j.PropertyConfigurator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.transwarp.cluster.comparison.ClusterComparisonImpl;
import io.transwarp.cluster.servlet.ClusterCheck;
import io.transwarp.common.GlobalArgs;
import io.transwarp.common.MyProperty;
import io.transwarp.common.TimeStatistics;
import io.transwarp.common.bean.InspectionData;
import io.transwarp.common.bean.OutputSelectBean;
import io.transwarp.common.bean.service.ServiceBean;
import io.transwarp.common.util.ExecShell;
import io.transwarp.common.util.HdfsConnUtils;
import io.transwarp.common.util.MyThreadPool;
import io.transwarp.common.util.UploadReport;
import io.transwarp.common.util.UtilTools;
import io.transwarp.hdfs.comparison.HdfsComparisonImpl;
import io.transwarp.hdfs.servlet.HdfsCheck;
import io.transwarp.node.comparison.NodeComparisonImpl;
import io.transwarp.node.servlet.NodeCheck;
import io.transwarp.report.comparison.Comparison;
import io.transwarp.report.comparison.ComparisonEnum;
import io.transwarp.report.comparison.ComparisonResult;
import io.transwarp.report.xls.Report;
import io.transwarp.service.comparison.ServiceComparisonImpl;
import io.transwarp.service.servlet.ServiceCheck;
import io.transwarp.table.comparison.TableComparisonImpl;
import io.transwarp.table.holodesk.servlet.ZkConnUtils;
import io.transwarp.table.servlet.TableCheck;

public final class Inspection {

	private final Logger log = LoggerFactory.getLogger(Inspection.class);
	
	private final InspectionData data;
	private final ComparisonResult comparisonResult;
	private final MyProperty prop_env;
	private enum checkItem {
		CLUSTER,
		NODE,
		SERVICE,
		HDFS,
		TABLE
	}
	
	public Inspection() {
		data = new InspectionData();
		comparisonResult = new ComparisonResult();
		prop_env = new MyProperty();
		init();
	}
	
	public void run() {
		SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		GlobalArgs.startTime = dateFormat.format(System.currentTimeMillis());
		for (checkItem item : checkItem.values()) {
			CheckInterface check = createCheckItem(item);
			try {
				check.check();
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		int waitTime = Integer.valueOf(prop_env.getProperty("waitTime", "0"));
		MyThreadPool.closeWhenAllCompleted(10000, waitTime*6);
		comparisonResult();
		closeAllConn();
		GlobalArgs.endTime = dateFormat.format(System.currentTimeMillis());
		outputReport();
		outputCostTime();
		try {
			UtilTools.compressTarGz(GlobalArgs.goalPath + "serviceConfigs");
			UtilTools.deleteFileOrDir(GlobalArgs.goalPath + "serviceConfigs");
		} catch (Exception e) {
			e.printStackTrace();
		}
		try {
			ExecShell.close();
		} catch (Exception e) {
			log.error("close shell pool is faild | {}", e.getMessage());
		}
		rebuildFile("completed");
		System.out.println("############################################");
		System.out.println("inspection is completed");
		System.out.println("############################################");
	}
	
	private CheckInterface createCheckItem(checkItem item) {
		switch (item) {
		case CLUSTER : {
			data.loadPropertyFile(GlobalArgs.configPath + "config/restapi/restapi.xml", 
					GlobalArgs.configPath + "config/restapi/loadMetric.xml");
			GlobalArgs.timeStatisticses.put(CheckItemEnum.CLUSTER.name(), new TimeStatistics());
			return new ClusterCheck(data);
		}
		
		case NODE : {
			String osType = prop_env.getProperty("os", "").toLowerCase(Locale.getDefault());
			String path_nodeCheck = GlobalArgs.configPath + "config/nodeCheck/nodeCheck_centos.xml";
			if (osType.startsWith("suse") || osType.startsWith("sles")) {
				path_nodeCheck = GlobalArgs.configPath + "config/nodeCheck/nodeCheck_suse.xml";
			}
			data.nodeInfo.loadPropertyFile(path_nodeCheck, GlobalArgs.configPath + "config/nodeCheck/portCheck.xml");
			GlobalArgs.timeStatisticses.put(CheckItemEnum.NODE.name(), new TimeStatistics());
			return new NodeCheck(data.nodeInfo);
		}
		
		case SERVICE : {
			if (GlobalArgs.version.startsWith("4")) {
				data.clusterInfo.loadPropertyFile(GlobalArgs.configPath + "config/serviceCheck/configNames_4x.properties",
						GlobalArgs.configPath + "config/serviceCheck/processCheck.xml", 
						GlobalArgs.configPath + "config/serviceCheck/serviceCheck_4x.xml",
						GlobalArgs.configPath + "config/hdfsCheck/hdfsCheck.xml");
			} else {
				data.clusterInfo.loadPropertyFile(GlobalArgs.configPath + "config/serviceCheck/configNames_5x.properties",
						GlobalArgs.configPath + "config/serviceCheck/processCheck.xml", 
						GlobalArgs.configPath + "config/serviceCheck/serviceCheck_5x.xml",
						GlobalArgs.configPath + "config/hdfsCheck/hdfsCheck.xml");
			}
			GlobalArgs.timeStatisticses.put(CheckItemEnum.SERVICE.name(), new TimeStatistics());
			return new ServiceCheck(data.clusterInfo);
		}
		
		case HDFS : {
			for (Entry<String, ServiceBean> entry : data.clusterInfo.getServices().entrySet()) {
				ServiceBean service = entry.getValue();
				if (service.getType().matches("\\S*HDFS\\S*") && !service.getType().equals("DOWN")) {
					GlobalArgs.hdfsPools.put(service.getName(), new HdfsConnUtils());
					return new HdfsCheck(data.clusterInfo, service);
				}
			}
		}
		
		case TABLE : {
			data.tableInfo.loadPropertyFile(GlobalArgs.configPath + "config/table/esCheck.xml");
			GlobalArgs.timeStatisticses.put(CheckItemEnum.TABLE.name(), new TimeStatistics());
			return new TableCheck(data.tableInfo, data.clusterInfo.getServices());
		}
		
		default : throw new RuntimeException("no completed this item");
		}
	}
	
	private void init() {
		PropertyConfigurator.configure(GlobalArgs.configPath + "config/log4j.properties");
		prop_env.load(GlobalArgs.configPath + "config/env.properties");
		GlobalArgs.prop_rules.load(GlobalArgs.configPath + "config/rules.xml");
		GlobalArgs.goalPath = prop_env.getProperty("goalPath", "/tmp/report/").trim();
		GlobalArgs.outputSheets = prop_env.getProperty("outputSheet", "version,roleMap").split(",");
		String encryptTablename = prop_env.getProperty("encryptTablename", "false");
		GlobalArgs.threadNumber = Integer.valueOf(prop_env.getProperty("threadNumber", "10"));
		GlobalArgs.managerIP = prop_env.getProperty("managerIP", "");
		data.setManagerIP(GlobalArgs.managerIP);
		data.setUsername(prop_env.getProperty("username", "guest"));
		data.setPassword(prop_env.getProperty("password", "guest"));
		data.setManagerPort(prop_env.getProperty("managerPort", "8180"));
		String nodeUser = prop_env.getProperty("nodeUser", "root");
		String rsa = prop_env.getProperty("userRsa", "/etc/transwarp/transwarp-id_rsa");
		String port = prop_env.getProperty("nodePort", "22");
		
		data.clusterInfo.setSidPath(prop_env.getProperty("serviceConfig", "/var/lib/transwarp-manager/master/data/data/Service.json"));
		
		if (!GlobalArgs.goalPath.endsWith("/")) {
			GlobalArgs.goalPath += "/";
		}
		ExecShell.init(nodeUser, rsa, port);
		MyThreadPool.init(GlobalArgs.threadNumber);
		data.nodeInfo.setPort(Integer.valueOf(port));
		data.tableInfo.setEncrypt(encryptTablename.equals("true") ? true : false);
		GlobalArgs.checkSelect = new OutputSelectBean(GlobalArgs.outputSheets);
		GlobalArgs.timeStamp = new SimpleDateFormat("yyyy-MM-dd").format(System.currentTimeMillis());
		
		UtilTools.buildDir(GlobalArgs.goalPath);
		UtilTools.deleteFileOrDir(GlobalArgs.goalPath + "completed");
		UtilTools.deleteFileOrDir(GlobalArgs.goalPath + "serviceConfigs");
		rebuildFile("started");
				
	}
	
	
	
	private void rebuildFile(String filename) {
		String filepath = GlobalArgs.goalPath + filename;
		File file = new File(filepath);
		if (file.exists()) {
			file.delete();
		}
		try {
			file.createNewFile();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	
	private void closeAllConn() {		
		for (Entry<String, HdfsConnUtils> entry : GlobalArgs.hdfsPools.entrySet()) {
			HdfsConnUtils hdfsPool = entry.getValue();
			try {
				hdfsPool.closeHdfsPool();
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		
		try {
			ZkConnUtils.close();
		} catch (Exception e) {
			log.error("close zookeeper connected is faild | {}", e.getMessage());
		}
	}
	
	private void comparisonResult() {
		Comparison check = null;
		for (ComparisonEnum checkItem : ComparisonEnum.values()) {
			log.info("comparison result of {}", checkItem);
			check = buildComparison(checkItem);
			if (check != null) {
				try {
					check.comparison();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
	}
	
	private Comparison buildComparison(ComparisonEnum itemName) {
		switch (itemName) {
		case CLUSTER : return new ClusterComparisonImpl(comparisonResult.CLUSTER, data);
		case NODE : return new NodeComparisonImpl(comparisonResult.NODE, data.nodeInfo, data.prop_metric);
		case SERVICE : return new ServiceComparisonImpl(comparisonResult, data.clusterInfo);
		case HDFS : return new HdfsComparisonImpl(comparisonResult.HDFS, data.clusterInfo);
		case TABLE : return new TableComparisonImpl(comparisonResult.TABLE, data.tableInfo);
		default : throw new RuntimeException("no find this check item : " + itemName.name());
		}
	}
	
	private String outputReport() {
		Report report = new Report(data, comparisonResult);
		SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
		String timeStamp = dateFormat.format(System.currentTimeMillis());
		String isUpload = prop_env.getProperty("isUpload", "false").trim();
		String reportFilename = "";
		String uploadUser = "";
		if (isUpload.equals("true")) {
			uploadUser = prop_env.getProperty("uploadUsername", "").trim();
			reportFilename = String.format("%s%s-REPORT-%s.xls", GlobalArgs.goalPath, uploadUser, timeStamp);
		} else {
			reportFilename = String.format("%sREPORT-%s.xls", GlobalArgs.goalPath, timeStamp);
		}
		report.outputExcel(reportFilename);
		if (prop_env.getProperty("summaryReport", "true").equals("true")) {
			report.outputHtml(GlobalArgs.goalPath, timeStamp);
		}
		if (isUpload.equals("true")) {
			uploadReport(uploadUser, reportFilename);
		}
		return reportFilename;
	}
	
	private void uploadReport(final String uploadUser, final String filename) {
		String isUseProxy = prop_env.getProperty("isUseProxy", "false").trim();
		UploadReport.Build build = UploadReport.custom()
			.setUploadUrl(prop_env.getProperty("uploadUrl", "").trim())
			.setUploadUsername(uploadUser)
			.setUploadPassword(prop_env.getProperty("uploadPassword", "").trim());
		if (isUseProxy.equals("true")) {
			build = build.setProxyIP(prop_env.getProperty("proxyUrl", "").trim())
				.setProxyPort(prop_env.getProperty("proxyPort", "").trim())
				.setProxyUsername(prop_env.getProperty("proxyUsername", "").trim())
				.setProxyPassword(prop_env.getProperty("proxyPassword", "").trim());
		}
		UploadReport upload = build.build();
		try {
			String result = upload.uploadFile(filename);
			log.info("upload report file result : [{}]", result);
		} catch (Exception e) {
			log.error("upload report faild");
			e.printStackTrace();
		}
	}
	
	private void outputCostTime() {
		for (Entry<String, TimeStatistics> entry : GlobalArgs.timeStatisticses.entrySet()) {
			String name = entry.getKey();
			TimeStatistics timeStatistics = entry.getValue();
			log.info("checkItem : {} cost time : {}", name, timeStatistics.getCostTime());
		}
	}
}
