package io.transwarp.node.servlet;

import java.io.File;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.transwarp.common.GlobalArgs;
import io.transwarp.common.bean.node.NodeBean;
import io.transwarp.common.util.AbstractRunnable;
import io.transwarp.common.util.ExecShell;
import io.transwarp.common.util.UtilTools;
import io.transwarp.inspection.CheckItemEnum;

public class MysqlSynchronizedRunnable extends AbstractRunnable {

	private final Logger log = LoggerFactory.getLogger(MysqlSynchronizedRunnable.class);
	private final String MASTER_SLAVE_SYNC = "mysql -e \"show slave status\\\\G\"";
	private final NodeBean node;
	private final String savePath;
	
	public MysqlSynchronizedRunnable(final NodeBean node) {
		this.node = node;
		this.savePath = String.format("%sserviceConfigs/mysql/%s/", GlobalArgs.goalPath, node.getIpAddress());
	}
	
	@Override
	public void startCheck() throws Exception {
		if (!GlobalArgs.version.startsWith("4")) {
			log.info("TDH version is [{}], no check mysql synchronized", GlobalArgs.version);
			return;
		}
		log.info("begin check mysql synchronized of TDH 4.x in [{}]", node.getHostName());
		UtilTools.buildDir(savePath);
		ExecShell.scpFileOfSudo(savePath, node.getIpAddress(), "/etc/", "my.cnf", "/tmp/inspection-temporary/mysql-config/");
		analysisMysqlSynch(savePath + "my.cnf");
	}
	
	@Override
	public void dealWithError(Exception e) {
		String errorMessage = String.format("mysql synchronized check of TDH 4.x in [%s] is faild, error message [%s]", node.getHostName(), e.getMessage());
		GlobalArgs.ERROR_INFO.add(errorMessage);
		log.error(errorMessage);
		e.printStackTrace();		
	}
	
	@Override
	public void closeThread() {
		log.info("mysql synchronized check in [{}] completed", node.getHostName());
		GlobalArgs.timeStatisticses.get(CheckItemEnum.NODE.name()).setEndTime();
	}
	
	private void analysisMysqlSynch(String path) throws Exception {
		File file = new File(path);
		if (!file.exists()) {
			node.setMysqlSynch("");
			return;
		}
		String fileValue = UtilTools.readFile(path);
		String[] lines = fileValue.split("\n");
		boolean hasMysql = false;
		boolean isSlave = false;
		for (String line : lines) {
			if (line.indexOf("[client]") != -1) {
				hasMysql = true;
			}
			if (line.startsWith("read_only")) {
				isSlave = true;
			}
		}
		if (!hasMysql) {
			node.setMysqlSynch("");
		} else if(!isSlave) {
			node.setMysqlSynch("active");
		} else {
			StringBuffer result = new StringBuffer();
			result.append("slave");
			try {
				String buffer = ExecShell.exec(node.getIpAddress(), MASTER_SLAVE_SYNC);
				result.append(":").append(isSynchronized(buffer));
			} catch (Exception e) {
				e.printStackTrace();
			} finally {
				node.setMysqlSynch(result.toString());
			}
		}
	
	}
	
	private String isSynchronized(String result) throws Exception {
		if (result == null) {
			return "true";
		}
		String[] lines = result.split("\n");
		for (String line : lines) {
			String[] items = line.split(":");
			if (items.length < 2) {
				continue;
			}
			String name = items[0].trim();
			String value = items[1].trim();
			if ((name.equals("Slave_IO_Running") || name.equals("Slave_SQL_Running")) && !value.equalsIgnoreCase("Yes")) {
				return "false";
			}
			if ((name.equals("Last_IO_Errno") || name.equals("Last_SQL_Errno")) && !value.equalsIgnoreCase("0")) {
				return "false";
			}
		}
		return "true";
	}
}
