package io.transwarp.report.html;

import java.io.BufferedWriter;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.List;

import io.transwarp.common.GlobalArgs;
import io.transwarp.common.bean.InspectionData;
import io.transwarp.report.comparison.ComparisonResult;
import io.transwarp.report.html.baseEnv.BaseEnvSummaryContentImpl;
import io.transwarp.report.html.cluster.ClusterSummaryContentImpl;
import io.transwarp.report.html.hdfs.HdfsSummaryContentImpl;
import io.transwarp.report.html.resource.ResourceSummaryContentImpl;
import io.transwarp.report.html.table.TableSummaryContentImpl;

public class SummaryReport {

	private final InspectionData data;
	private final ComparisonResult comparisonResult;
	
	private BufferedWriter writer;
	private String[] checkItems = new String[]{"baseEnv|基础环境检查", "cluster|集群稳定性检查", "hdfsCheck|文件分布式系统检查", "resourceCheck|资源分配检查", "tableCheck|数据表检查"};
	
	private List<String> summaryTotalInfo;
	private List<String> summaryIndex;
	private List<String> summaryBody;
	
	public SummaryReport(final InspectionData data, final ComparisonResult comparisonResult) {
		this.data = data;
		this.comparisonResult = comparisonResult;
		this.summaryTotalInfo = new ArrayList<String>();
		this.summaryIndex = new ArrayList<String>();
		this.summaryBody = new ArrayList<String>();
	}
	
	public void outputReport(final String path) {
		openWriter(path);
		try {
			writeHtmlHead();
			writeContext();
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			closeWriter();			
		}
	}
	
	private void openWriter(final String path) {
		try {
			FileOutputStream outputStream = new FileOutputStream(path);
			writer = new BufferedWriter(new OutputStreamWriter(outputStream, "UTF-8"));
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	private void writeHtmlHead() throws Exception {
		writer.write("<!DOCTYPE HTML>\n");
		writer.write("<html> <head>\n");
		writer.write("  <meta charset=\"utf-8\">\n");
		writer.write("  <link rel=\"stylesheet\" href=\"https://cdn.bootcss.com/bootstrap/3.3.7/css/bootstrap.min.css\">\n");
		writer.write("  <title>巡检总结报告</title>\n");
		writer.write("  <script src=\"https://cdn.bootcss.com/jquery/2.1.1/jquery.min.js\"></script>\n");
		writer.write("  <script src=\"https://cdn.bootcss.com/bootstrap/3.3.7/js/bootstrap.min.js\"></script>\n");
		setHtmlStyle();
		writer.write("  <div align=\"center\"> <h1>集群安全检查</h1> <div class=\"bline\"/>\n");
		writer.write("    <div id=\"summaryWithIndex\" class=\"titleStyle\"> ");
		summaryTotalInfo.add("<div id=\"summary\"> <h2 class=\"checkItem\">综述</h2>\n");
		summaryTotalInfo.add("      <table style=\"word-wrap: break-work; overflow: hidden;\">\n");
		summaryTotalInfo.add("        <tr> <th width=\"25%\">任务名称</th> <td width=\"40%\">集群健康检查</td> </tr>\n");
		summaryTotalInfo.add(String.format("        <tr> <th width=\"25%%\">集群版本号</th> <td width=\"40%%\">%s</td></tr>\n", GlobalArgs.version));
		summaryTotalInfo.add(String.format("        <tr> <th width=\"25%%\">集群节点数</th> <td width=\"40%%\">%s</td></tr>\n", data.nodeInfo.getNodes().size()));
		summaryTotalInfo.add(String.format("        <tr> <th width=\"25%%\">巡检时间</th> <td width=\"40%%\">开始：%s</td></tr>\n", GlobalArgs.startTime));
		summaryTotalInfo.add(String.format("        <tr> <th width=\"25%%\"/> <td width=\"40%%\">结束：%s</td></tr>\n", GlobalArgs.endTime));
	}
	
	private void setHtmlStyle() throws Exception {
		writer.write("  <style type=\"text/css\">\n");
		writer.write("    html, body { text-align:center; overflow:visible; +overflow:auto }\n");
		writer.write("    .bline { padding-top:5px; border-top-width:3px; border-top-style:solid; border-top-color:#bababa; width:80%; }\n");
		writer.write("    .showtable { width:100%; border-collapse:collapse; border-padding:0px; }\n");
		writer.write("    .showtable tr th { background:#d9d9d9; text-align:left; height:20px; border-bottom:1px #bababa solid; padding-left:15px; white-space:normal; word-break:break-all; }\n");
		writer.write("    .showtable tr td { text-align:left; border-bottom:1px #bababa solid; padding-left:15px; white-space:normal; word-break:break-all; }\n");
		writer.write("    .checkItem { cursor:pointer; background:#bababa; font-weight:bold; line-height:25px; font-size:16px; padding:1px 0px 1px 1px; background-repeat:no-repeat; background-position:right; margin:1px 0 0; }\n");
		writer.write("    .atomItem { cursor:pointer;font-size:14px;text-align:left;padding-left:15px;border-bottom:2px #bababa solid; background-repeat: no-repeat; background-position: right;}\n");
		writer.write("    .titleStyle { width:100%; clear:both; text-align:left; }\n");
		writer.write("    #reportIndex { border:1px solid #bababa; width:25%; text-align:left; float:right; padding-bottom:10px; }\n");
		writer.write("    #reportIndex h2 { line-height:25px; font-size:14px; padding-left:5px; }\n");
		writer.write("    #reportIndex ul { list-style-type:none; margin:0px; list-style-position:outside; padding-left:10px; position:relative; }\n");
		writer.write("    #reportIndex ul ol { list-style-type:none; margin:0px; list-style-position:inside; padding-left:25px; }\n");
		writer.write("    #summary { float:left; width:75%; }\n");
		writer.write("  </style> </head> <body>\n");
	}
	
	private void writeContext() throws Exception {
		summaryIndex.add("<div id=\"reportIndex\"> <h2>目录</h2> <ul id=\"titleList\">\n");
		summaryIndex.add("  <li><a href=\"#summary\">综述</a></li>\n");
		int number = checkItems.length;
//		int risk = 0;
		for (int i = 0; i < number; i++) {
			String[] items = checkItems[i].split("\\|");
			AbstractSummaryContent content = createItem(items[0]);
			if (content != null) {
				content.buildHtml(items[0], items[1]);
				summaryIndex.add(String.format("  <li><a href=\"#%s\">%s</a></li>\n", items[0], items[1]));
				summaryBody.addAll(content.getResult());
				summaryBody.add("<br> <br>\n");
				String firstCol = i == 0 ? "<th>检查项</th>" : "<th/>";
				summaryTotalInfo.add(String.format("    <tr> %s <td width=\"30%%\">%s</td> <td style=\"color:green\" width=\"15%%\">正常：%s</td> <td width=\"10%%\" /> <td style=\"color:red\" width=\"15%%\">告警：%s</td> </tr>\n", 
						firstCol, items[1], content.getSuccessNumber(), content.getFailureNumber()));
				addIndex(items[0], content.getCheckItems());
//				risk += content.getRisk();
			}
		}
//		summaryTotalInfo.add(String.format("    <tr> <th>集群风险值</th> <td>%s</td> </tr>\n", risk));
		summaryTotalInfo.add("  </table> </div>");
		summaryIndex.add("  </ul> </div>\n");
		for (String line : summaryIndex) {
			writer.write(line);
		}
		for (String line : summaryTotalInfo) {
			writer.write(line);
		}
		writer.write("</div>\n");
		for (String line : summaryBody) {
			writer.write(line);
		}
		writer.write("</div> <br> </body> </html>");
	}
	
	private void addIndex(String itemName_EN, List<String> contentItems) {
		summaryIndex.add("  <ol>\n");
		int number = contentItems.size();
		for (int i = 0; i < number; i++) {
			summaryIndex.add(String.format("    <li><a href=\"#%s%03d\">%s</a></li>\n", itemName_EN, i, contentItems.get(i)));
		}
		summaryIndex.add("  </ol>\n");
	}
	
	private AbstractSummaryContent createItem(String itemName) {
		if (itemName.equals("baseEnv")) {
			return new BaseEnvSummaryContentImpl(data, comparisonResult);
		} else if (itemName.equals("cluster")) {
			return new ClusterSummaryContentImpl(data, comparisonResult);
		} else if (itemName.equals("hdfsCheck")) {
			return new HdfsSummaryContentImpl(data, comparisonResult);
		} else if (itemName.equals("tableCheck")) {
			return new TableSummaryContentImpl(data, comparisonResult);
		} else if (itemName.equals("resourceCheck")) {
			return new ResourceSummaryContentImpl(data, comparisonResult);
		}
		return null;
	}
	
	private void closeWriter() {
		try {
			if (writer == null) {
				return;
			}
			writer.flush();
			writer.close();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
}
