package io.transwarp.report.xls;

import java.util.Locale;

import jxl.write.WritableWorkbook;
import io.transwarp.cluster.report.AlertXlsReportImpl;
import io.transwarp.cluster.report.LicenseXlsReportImpl;
import io.transwarp.cluster.report.RoleMapXlsReportImpl;
import io.transwarp.cluster.report.VersionXlsReportImpl;
import io.transwarp.common.GlobalArgs;
import io.transwarp.common.bean.*;
import io.transwarp.common.util.ExecShell;
import io.transwarp.common.util.UtilTools;
import io.transwarp.common.util.WorkbookUtilTools;
import io.transwarp.hdfs.report.HdfsInfoXlsReportImpl;
import io.transwarp.hdfs.report.HdfsSpaceXlsReportImpl;
import io.transwarp.node.report.DiskXlsReportImpl;
import io.transwarp.node.report.MetricXlsReportImpl;
import io.transwarp.node.report.NodeXlsReportImpl;
import io.transwarp.node.report.PortXlsReportImpl;
import io.transwarp.report.comparison.ComparisonResult;
import io.transwarp.report.html.SummaryReport;
import io.transwarp.service.kafka.report.KafkaXlsReportImpl;
import io.transwarp.service.process.report.ProcessXlsReportImpl;
import io.transwarp.table.es.report.ESParamXlsReportImpl;
import io.transwarp.table.es.report.ESTableXlsReportImpl;
import io.transwarp.table.hbase.report.HBaseXlsReportImpl;
import io.transwarp.table.holodesk.report.HolodeskXlsReportImpl;
import io.transwarp.table.orc.report.OrcXlsReportImpl;
import io.transwarp.table.report.EncryptTableXlsReportImpl;
import io.transwarp.table.servlet.TableTypeEnum;
import io.transwarp.table.text.report.OtherTableXlsReportImpl;
import io.transwarp.table.text.report.TextXlsReportImpl;

public class Report {

	private final InspectionData data;
	private final ComparisonResult comparisonResult;
	
	public Report(final InspectionData data,
			final ComparisonResult comparisonResult) {
		this.data = data;
		this.comparisonResult = comparisonResult;
	}
	
	public void outputExcel(final String path) {
		outputReport(path, GlobalArgs.outputSheets);
		if (data.tableInfo.isEncryptTablename()) {
			outputEncryptTable(GlobalArgs.goalPath + "EncryptTable.xls");
		}
	}
	
	public void outputHtml(String path, String timeStamp) {
		String outputPath = String.format("%ssummaryReport-%s.html", path, timeStamp);
		UtilTools.deleteFileOrDir(outputPath);
		SummaryReport summaryReport = new SummaryReport(data, comparisonResult);
		summaryReport.outputReport(outputPath);
		String lnFile = path + "summaryReport.html";
		UtilTools.deleteFileOrDir(lnFile);
		ExecShell.exec(String.format("ln -s %s %s", outputPath, lnFile));
	}
	
	private void outputReport(final String path, final String[] outputItems) {
		WritableWorkbook workbook = null;
		try {
			workbook = WorkbookUtilTools.buildExcel(path);
			XlsReport reportItem = null;
			if (GlobalArgs.ERROR_INFO.size() > 0) {
				reportItem = buildReportItem("error", workbook);
				reportItem.writeSheet();
			}
			reportItem = buildReportItem("alerts", workbook);
			reportItem.writeSheet();
			for(String outputItem : outputItems) {
				try {
					reportItem = buildReportItem(outputItem, workbook);
					reportItem.writeSheet();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
			if (data.tableInfo.hasOtherTable()) {
				reportItem = buildReportItem("ddl_other", workbook);
				reportItem.writeSheet();
			}
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			WorkbookUtilTools.close(workbook);
		}
	}
	
	private void outputEncryptTable(final String path) {
		WritableWorkbook workbook = null;
		try {
			workbook = WorkbookUtilTools.buildExcel(path);
			XlsReport reportItem = null;
			for (TableTypeEnum tableType : TableTypeEnum.values()) {
				try {
					reportItem = buildTableEncrypt(tableType, workbook);
					reportItem.writeSheet();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			WorkbookUtilTools.close(workbook);
		}
	}
	
	private XlsReport buildReportItem(String item, WritableWorkbook workbook) {
		item = item.toLowerCase(Locale.getDefault());
		if (item.equals("error")) {
			return new ErrorXlsReportImpl(workbook, ReportEnum.ERROR);
		} else if (item.equals("alerts")) {
			return new AlertXlsReportImpl(workbook, ReportEnum.ALERTS, data.clusterInfo);
		} else if (item.equals("version")) {
			return new VersionXlsReportImpl(workbook, ReportEnum.VERSION, data.nodeInfo);
		} else if (item.equals("rolemap")) {
			return new RoleMapXlsReportImpl(workbook, ReportEnum.ROLEMAP, comparisonResult.CLUSTER);
		} else if (item.equals("nodeinfo")) {
			return new NodeXlsReportImpl(workbook, ReportEnum.NODEINFO, data.nodeInfo.getNodes());
		} else if (item.equals("disk")) {
			return new DiskXlsReportImpl(workbook, ReportEnum.DISK, data.nodeInfo.getNodes(), comparisonResult.NODE);
		} else if (item.equals("process")) {
			return new ProcessXlsReportImpl(workbook, ReportEnum.PROCESS, data.clusterInfo.getServices());
		} else if (item.equals("portinfo")) {
			return new PortXlsReportImpl(workbook, ReportEnum.PORTINFO, data.nodeInfo);
		} else if (item.equals("hdfsinfo")) {
			return new HdfsInfoXlsReportImpl(workbook, ReportEnum.HDFSINFO, data.clusterInfo.getHdfsMetaDatas());
		} else if (item.equals("hdfsspace")) {
			return new HdfsSpaceXlsReportImpl(workbook, ReportEnum.HDFSSPACE, comparisonResult.HDFS.hdfsSpace);
		} else if (item.equals("license")) {
			return new LicenseXlsReportImpl(workbook, ReportEnum.LICENSE, data.clusterInfo.getLicense());
		} else if (item.equals("metric")) {
			return new MetricXlsReportImpl(workbook, ReportEnum.METRIC, comparisonResult.NODE);
		} else if (item.equals("ddl_text")) {
			return new TextXlsReportImpl(workbook, ReportEnum.DDL_TEXT, data.tableInfo, comparisonResult.TABLE);
		} else if (item.equals("ddl_orc")) {
			return new OrcXlsReportImpl(workbook, ReportEnum.DDL_ORC, data.tableInfo, comparisonResult.TABLE);
		} else if (item.equals("ddl_hbase")) {
			return new HBaseXlsReportImpl(workbook, ReportEnum.DDL_HBASE, data.tableInfo, comparisonResult.TABLE);
		} else if (item.equals("ddl_holodesk")) {
			return new HolodeskXlsReportImpl(workbook, ReportEnum.DDL_HOLODESK, data.tableInfo, comparisonResult.TABLE);
		} else if (item.equals("ddl_es")) {
			return new ESTableXlsReportImpl(workbook, ReportEnum.DDL_ES, data.tableInfo, comparisonResult.TABLE);
		} else if (item.equals("servicecheck_es")) {
			return new ESParamXlsReportImpl(workbook, ReportEnum.SERVICECHECK_ES, data.tableInfo);
		} else if (item.equals("ddl_other")) {
			return new OtherTableXlsReportImpl(workbook, ReportEnum.DDL_OTHER, data.tableInfo);
		} else if (item.equals("servicecheck_kafka")) {
			return new KafkaXlsReportImpl(workbook, ReportEnum.SERVICECHECK_KAFKA, data.clusterInfo.getKafkaTopicMaps(), comparisonResult.SERVICE);
		} else if (item.equals("costTime")) {
			return new CostTimeXlsReportImpl(workbook, ReportEnum.COST_TIME);
		}
		throw new RuntimeException("no find this item : " + item);
	}
	
	private XlsReport buildTableEncrypt(TableTypeEnum tableType, WritableWorkbook workbook) {
		switch(tableType) {
		case TEXT : return new EncryptTableXlsReportImpl(workbook, TableTypeEnum.TEXT, 
				new String[]{"table id", "database name", "table name"}, data.tableInfo.TABLE_TEXT.getEncryptText());
		case ORC : return new EncryptTableXlsReportImpl(workbook, TableTypeEnum.ORC, 
				new String[]{"table id", "database name", "table name"}, data.tableInfo.TABLE_ORC.getEncryptOrc());
		case HBASE : return new EncryptTableXlsReportImpl(workbook, TableTypeEnum.HBASE,
				new String[]{"table id", "database name", "table name", "hbase name"}, data.tableInfo.TABLE_HBASE.getEncryptHBase());
		case HOLODESK : return new EncryptTableXlsReportImpl(workbook, TableTypeEnum.HOLODESK,
				new String[]{"table id", "database name", "table name"}, data.tableInfo.TABLE_HOLODESK.getEncryptHolodesk());
		case ES : return new EncryptTableXlsReportImpl(workbook, TableTypeEnum.ES, 
				new String[]{"table id", "database name", "table name", "es name"}, data.tableInfo.TABLE_ES.getEncryptES());
		default : throw new RuntimeException("no find this table type : " + tableType);
		}
	}
}
