package io.transwarp.service.comparison;

import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import io.transwarp.common.GlobalArgs;
import io.transwarp.common.bean.node.NodeBean;
import io.transwarp.common.bean.service.RoleBean;
import io.transwarp.common.bean.service.ServiceBean;
import io.transwarp.common.util.UtilTools;
import io.transwarp.report.comparison.Comparison;

public class ExecutorComparisonImpl implements Comparison {
	
	private final ComparisonResultOfService comparisonResult;
	private final Map<String, ServiceBean> services;
	private Double coreUsedLimit;
	private Double coreNumberLimit;
	private Double memoryUsedLimit;
	private Double memorySizeLimit;
	private Boolean warning;
	
	public ExecutorComparisonImpl(ComparisonResultOfService comparisonResult,
			Map<String, ServiceBean> services) {
		this.comparisonResult = comparisonResult;
		this.services = services;
		coreUsedLimit = Double.valueOf(GlobalArgs.prop_rules.getProperty("executor.core.usedPercent.limit", "0.5"));
		coreNumberLimit = Double.valueOf(GlobalArgs.prop_rules.getProperty("executor.core.number.limit", "12"));
		memoryUsedLimit = Double.valueOf(GlobalArgs.prop_rules.getProperty("executor.memory.usedPercent.limit", "0.5"));
		memorySizeLimit = Double.valueOf(GlobalArgs.prop_rules.getProperty("executor.memory.size.limit", "30"))*1024*1024*1024;
		warning = false;
	}

	@Override
	public void comparison() throws Exception {
		for (Entry<String, ServiceBean> entry : services.entrySet()) {
			ServiceBean service = entry.getValue();
			String serviceType = service.getType();
			if (serviceType.equals("INCEPTOR_SQL") || serviceType.equals("INCEPTOR") || serviceType.equals("SLIPSTREAM")) {
				comparisonExecutor(service);
			}
		}
	}
	
	private void comparisonExecutor(ServiceBean service) {
		String servicename = service.getName();
		List<RoleBean> roles = service.getRoles();
		for (RoleBean role : roles) {
			try {
				if (!role.getRoleType().equals("INCEPTOR_EXECUTOR")) {
					continue;
				}
				String[] result = new String[8];
				NodeBean node = role.getNode();
				String hostname = node.getHostName();
				Double totalCores = UtilTools.getValueFromSize(node.getCpu().split(" ")[0]);
				Double totalMemory = Double.valueOf(node.getTotalPhysMemBytes());
				Double executorCore = 0.0;
				Double executorMemory = 0.0;
				Integer activeExecutor = service.getActiveExecutorByHost(hostname);
				if (activeExecutor == null) {
					activeExecutor = 0;
				}
				String resource = role.getResource();
				String[] items = resource.split(",");
				for (String item : items) {
					if (item.trim().equals("")) {
						continue;
					}
					String value = item.split(": ")[1];
					if (item.startsWith("memory")) {
						executorMemory = activeExecutor*UtilTools.getValueFromSize(value);
					} else if (item.startsWith("core")) {
						executorCore = activeExecutor*UtilTools.getValueFromSize(value);
					}
				}
				if ((totalCores != 0 && executorCore/totalCores > coreUsedLimit) || executorCore > coreNumberLimit) {
					warning = true;
				} 
				if ((totalMemory != 0 && executorMemory/totalMemory > memoryUsedLimit) || executorMemory > memorySizeLimit) {
					warning = true;
				}
				result[0] = String.format("%s-%s", servicename, role.getName());
				result[1] = hostname;
				result[2] = String.format("%.0f", executorCore);
				result[3] = String.format("%.0f", totalCores);
				result[4] = String.format("%.2f %%", 100*executorCore/totalCores);
				result[5] = String.format("%s", UtilTools.getCarrySize(executorMemory.longValue()));
				result[6] = String.format("%s", UtilTools.getCarrySize(totalMemory.longValue()));
				result[7] = String.format("%.2f %%", 100*executorMemory/totalMemory);
				comparisonResult.executorInfos.add(result);
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		comparisonResult.executorWarn = warning;
	}

}
