package io.transwarp.service.comparison;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import io.transwarp.common.GlobalArgs;
import io.transwarp.common.bean.service.KafkaPartitionBean;
import io.transwarp.common.bean.service.KafkaTopicBean;
import io.transwarp.report.comparison.Comparison;

public class KafkaComparisonImpl implements Comparison {
	
	private final Map<String, Map<String, KafkaTopicBean>> allTopics;
	private final ComparisonResultOfService comparisonResult;
	
	private String leaderValueLimit;
	private int partitionNumberLimit;
	private int replicasNumberLimit;
	
	public KafkaComparisonImpl(final ComparisonResultOfService comparisonResult,
			final Map<String, Map<String, KafkaTopicBean>> allTopics) {
		this.comparisonResult = comparisonResult;
		this.allTopics = allTopics;
		this.leaderValueLimit = GlobalArgs.prop_rules.getProperty("kafka.topic.leader.value.limit", "-1");
		this.partitionNumberLimit = Integer.valueOf(GlobalArgs.prop_rules.getProperty("kafka.topic.partition.number.limit", "2"));
		this.replicasNumberLimit = Integer.valueOf(GlobalArgs.prop_rules.getProperty("kafka.topic.replica.number.limit", "2"));
	}

	@Override
	public void comparison() throws Exception {
		for (Entry<String, Map<String, KafkaTopicBean>> entry : allTopics.entrySet()) {
			String servicename = entry.getKey();
			Map<String, KafkaTopicBean> topics = entry.getValue();
			List<KafkaTopicBean> trueTopicInfos = new ArrayList<KafkaTopicBean>();
			List<KafkaTopicBean> errorTopicInfos = new ArrayList<KafkaTopicBean>();
			for (Entry<String, KafkaTopicBean> topicEntry : topics.entrySet()) {
				KafkaTopicBean topic = topicEntry.getValue();
				boolean warn = analysisTopic(topic);
				if (warn) {
					errorTopicInfos.add(topic);
				} else {
					trueTopicInfos.add(topic);
				}
			}
			comparisonResult.trueKafkaInfos.put(servicename, sortKafkaTopic(trueTopicInfos));
			comparisonResult.errorKafkaInfos.put(servicename, sortKafkaTopic(errorTopicInfos));
		}

	}
	
	private boolean analysisTopic(final KafkaTopicBean topic) {
		List<KafkaPartitionBean> partitions = topic.getPartitions();
		int number = partitions.size();
		if (number < partitionNumberLimit) {
			return true;
		}
		for (KafkaPartitionBean partition : partitions) {
			String leader = partition.getLeader();
			if (leader.equals(leaderValueLimit)) {
				return true;
			}
			String[] replicas = partition.getReplicas().split(",");
			if (replicas.length < replicasNumberLimit) {
				return true;
			}
		}
		return false;
	}
	
	private KafkaTopicBean[] sortKafkaTopic(List<KafkaTopicBean> topics) {
		int number = topics.size();
		if (number == 0) {
			return null;
		}
		KafkaTopicBean[] orderTopics = new KafkaTopicBean[number];
		orderTopics = topics.toArray(orderTopics);
		return orderTopics;
	}
}
