package io.transwarp.service.config.servlet;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.dom4j.Element;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.transwarp.common.GlobalArgs;
import io.transwarp.common.ReadXmlUtil;
import io.transwarp.common.bean.node.NodeBean;
import io.transwarp.common.bean.service.ConfigBean;
import io.transwarp.common.bean.service.RoleBean;
import io.transwarp.common.bean.service.ServiceBean;
import io.transwarp.common.util.AbstractRunnable;
import io.transwarp.common.util.ExecShell;
import io.transwarp.common.util.MyThreadPool;
import io.transwarp.common.util.UtilTools;
import io.transwarp.inspection.CheckItemEnum;

public class ConfigCheckRunnable extends AbstractRunnable {

	private final Logger log = LoggerFactory.getLogger(ConfigCheckRunnable.class);
	private final ServiceBean service;
	private final String[] configFiles;
	private final ReadXmlUtil prop_service_check;
	
	public ConfigCheckRunnable(final ServiceBean service,
			final String[] configFiles,
			final ReadXmlUtil prop_service_check) {
		this.service = service;
		this.configFiles = configFiles;
		this.prop_service_check = prop_service_check;
	}
	
	@Override
	public void startCheck() {
		final String servicename = service.getName();
		log.info("begin check config file of service [{}]", servicename);
		final ConfigBean config = new ConfigBean(servicename);
		Set<String> ips = getIps();
		String sid = service.getSid();
		for (String ip : ips) {
			String savePath = String.format("%sserviceConfigs/%s/%s/", GlobalArgs.goalPath, servicename, ip);
			String configPath = String.format("/etc/%s/conf/", sid);
			String bufferDir = String.format("%s%s-config/", GlobalArgs.temparoryPath, sid);
			UtilTools.buildDir(savePath);
			for (String configFile : configFiles) {
				try {
					ExecShell.scpFileOfSudo(savePath, ip, configPath, configFile, bufferDir);
					config.addConfigFile(ip, configFile, analysisFile(savePath + configFile));
				} catch (Exception e) {
					log.error("analysis config file of service [{}] is faild, error message is [{}]", servicename, e.getMessage());
					e.printStackTrace();
					continue;
				}
			}
		}
		service.setConfig(config);
		addResourceCheck(config);
		log.info("config check of service [{}] is completed", servicename);
	}
	
	@Override
	public void dealWithError(Exception e) {
		String errorMessage = String.format("check service config of service [%s] is faild | %s", service.getName(), e.getMessage());
		GlobalArgs.ERROR_INFO.add(errorMessage);
		log.error(errorMessage);
		e.printStackTrace();
		if (service.getConfig() == null) {
			service.setConfig(new ConfigBean(service.getName()));
		}
	}
	
	@Override
	public void closeThread() {
		log.info("config check of [{}] is completed", service.getName());
		GlobalArgs.timeStatisticses.get(CheckItemEnum.SERVICE.name()).setEndTime();
	}
	
	private Set<String> getIps() {
		Set<String> ips = new HashSet<String>();
		List<RoleBean> roles = service.getRoles();
		for (RoleBean role : roles) {
			NodeBean node = role.getNode();
			ips.add(node.getIpAddress());
		}
		return ips;
	}
	
	private Map<String, String> analysisFile(final String configPath) throws Exception {
		final Map<String, String> fileValues = new HashMap<String, String>();
		File file = new File(configPath);
		if (!file.exists()) {
			return fileValues;
		}
		try {
			if (configPath.endsWith(".xml")) {
				getValueOfXml(fileValues, configPath);
			} else if (configPath.endsWith(".sh") || configPath.endsWith("-env")) {
				getValueBySplit(fileValues, "=", configPath);
			} else if (configPath.endsWith(".yml")) {
				getValueBySplit(fileValues, ":", configPath);
			} else {
				throw new RuntimeException("type of config file " + configPath + " is no read");
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return fileValues;
	}
	
	private void getValueOfXml(final Map<String, String> configValue, final String configPath) throws Exception {
		ReadXmlUtil xmlRead = new ReadXmlUtil();
		xmlRead.load(configPath);
		List<Element> properties = xmlRead.getAll();
		for (Element property : properties) {
			String name = property.elementText("name");
			String value = property.elementText("value");
			configValue.put(name, value);
		}
	}
	
	private void getValueBySplit(final Map<String, String> configValue, String splitChar, final String configPath) throws Exception {
		BufferedReader reader = new BufferedReader(new FileReader(configPath));
		while (true) {
			String line = reader.readLine();
			if (line == null) {
				break;
			}
			int splitId = line.indexOf(splitChar, 1);
			if (splitId == -1 || line.startsWith("#")) {
				continue;
			}
			String name = line.substring(0, splitId);
			if (name.startsWith("export")) {
				name = name.substring(7);
			}
			name = name.trim();
			String value = line.substring(splitId + 1).trim();
			String defaultValue = configValue.get(name);
			if (defaultValue != null) {
				value = String.format("%s %s", defaultValue, value);
			}
			configValue.put(name, value);
		}
		reader.close();
	}
	
	private void addResourceCheck(ConfigBean config) {
		List<RoleBean> roles = service.getRoles();
		String serviceType = service.getType();
		for (RoleBean role : roles) {
			MyThreadPool.addNewThread(new ResourceCheckRunnable(serviceType, role, config, prop_service_check));
		}
		if (serviceType.equals("INCEPTOR") || serviceType.equals("INCEPTOR_SQL") || serviceType.equals("SLIPSTREAM")) {
			MyThreadPool.addNewThread(new ExecutorCheckRunnable(service));
		}
	}
}
