package io.transwarp.service.kafka.report;

import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import jxl.write.Label;
import jxl.write.WritableCellFormat;
import jxl.write.WritableSheet;
import jxl.write.WritableWorkbook;
import io.transwarp.common.bean.service.KafkaPartitionBean;
import io.transwarp.common.bean.service.KafkaTopicBean;
import io.transwarp.report.xls.ReportEnum;
import io.transwarp.report.xls.XlsReport;
import io.transwarp.service.comparison.ComparisonResultOfService;

public class KafkaXlsReportImpl implements XlsReport {

	private final WritableSheet sheet;
	private final ComparisonResultOfService comparisonResult;
	private final Map<String, Map<String, KafkaTopicBean>> allTopicInfos;
	private int maxCol;
	
	public KafkaXlsReportImpl(final WritableWorkbook workbook, 
			final ReportEnum checkItem,
			final Map<String, Map<String, KafkaTopicBean>> allTopicInfos,
			final ComparisonResultOfService comparisonResult) {
		this.comparisonResult = comparisonResult;
		this.allTopicInfos = allTopicInfos;
		this.sheet = workbook.createSheet("serviceCheck_kafka", checkItem.ordinal());
		this.maxCol = 0;
	}
	
	@Override
	public void writeSheet() throws Exception {
		int beginRow = 0;
		int totalTopics = 0;
		for (Entry<String, Map<String, KafkaTopicBean>> entry : allTopicInfos.entrySet()) {
			String servicename = entry.getKey();
			Map<String, KafkaTopicBean> topicInfos = entry.getValue();
			if (topicInfos.size() == 0) {
				continue;
			}
			totalTopics += topicInfos.size();
			sheet.addCell(new Label(0, beginRow++, servicename, CELL_BOLD));
			writeTitle(beginRow++);
			beginRow = writeAllTopics(beginRow, servicename);
		}
		if (totalTopics == 0) {
			sheet.addCell(new Label(0, 0, "no found topic", CELL_BOLD));
		} else {
			for (int i = 0; i < maxCol; i++) {
				sheet.setColumnView(i, 20);
			}
		}
	}

	private void writeTitle(final int row) throws Exception {
		int column = 1;
		sheet.addCell(new Label(column++, row, "topic", CELL_BOLD));
		sheet.addCell(new Label(column++, row, "ReplicationFactor", CELL_BOLD));
		sheet.addCell(new Label(column++, row, "configs", CELL_BOLD));
		sheet.addCell(new Label(column++, row, "partitionId", CELL_BOLD));
		sheet.addCell(new Label(column++, row, "leader", CELL_BOLD));
		sheet.addCell(new Label(column++, row, "replicas", CELL_BOLD));
		sheet.addCell(new Label(column++, row, "Isr", CELL_BOLD));
		this.maxCol = Math.max(column, maxCol);
	}
	
	private int writeAllTopics(int beginRow, final String servicename) throws Exception {
		// write error topic
		KafkaTopicBean[] errorTopics = comparisonResult.errorKafkaInfos.get(servicename);
		if (errorTopics != null) {
			for (KafkaTopicBean topic : errorTopics) {
				beginRow = writeTopic(beginRow, topic, CELL_ERROR);
			}
		}
		
		// write true topic
		KafkaTopicBean[] trueTopics = comparisonResult.trueKafkaInfos.get(servicename);
		if (trueTopics != null) {
			for (KafkaTopicBean topic : trueTopics) {
				beginRow = writeTopic(beginRow, topic, CELL_THIN);
			}
		}
		return beginRow + 1;
	}
	
	private int writeTopic(int beginRow, final KafkaTopicBean topic, WritableCellFormat cell) throws Exception {
		int column = 1;
		List<KafkaPartitionBean> partitions = topic.getPartitions();
		int partitionNumber = partitions.size();
		addTopicItem(column++, beginRow, topic.getTopic(), partitionNumber, cell);
		addTopicItem(column++, beginRow, topic.getReplicationFactor(), partitionNumber, cell);
		addTopicItem(column++, beginRow, topic.getConfigs(), partitionNumber, cell);
		if (partitionNumber == 0) {
			return beginRow;
		}
		KafkaPartitionBean[] orders = new KafkaPartitionBean[partitionNumber];
		orders = partitions.toArray(orders);
		try {
			Arrays.sort(orders, new Comparator<KafkaPartitionBean>() {
				@Override
				public int compare(KafkaPartitionBean par1, KafkaPartitionBean par2) {
					int id1 = Integer.valueOf(par1.getPartitionId());
					int id2 = Integer.valueOf(par2.getPartitionId());
					if (id1 > id2) {
						return 1;
					} else if (id1 == id2) {
						return 0;
					} else {
						return -1;
					}
				}
			});
		} catch (Exception e) {
			e.printStackTrace();
		}
		for (KafkaPartitionBean partition : orders) {
			int beginCol = column;
			sheet.addCell(new Label(beginCol++, beginRow, partition.getPartitionId(), cell));
			sheet.addCell(new Label(beginCol++, beginRow, partition.getLeader(), cell));
			sheet.addCell(new Label(beginCol++, beginRow, partition.getReplicas(), cell));
			sheet.addCell(new Label(beginCol++, beginRow, partition.getIsr(), cell));
			beginRow += 1;
		}
		return beginRow;
	}
	
	private void addTopicItem(final int column, final int row, final String value, final int high, WritableCellFormat cell) throws Exception {
		if (high > 1) {
			sheet.mergeCells(column, row, column, row + high - 1);
		}
		sheet.addCell(new Label(column, row, value, cell));
	}
}
