package io.transwarp.service.servlet;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;

import io.transwarp.common.GlobalArgs;
import io.transwarp.common.bean.ClusterInfo;
import io.transwarp.common.bean.service.KafkaTopicBean;
import io.transwarp.common.bean.service.ProcessBean;
import io.transwarp.common.bean.service.RoleBean;
import io.transwarp.common.bean.service.ServiceBean;
import io.transwarp.common.util.MyThreadPool;
import io.transwarp.inspection.CheckInterface;
import io.transwarp.service.config.servlet.ConfigCheckRunnable;
import io.transwarp.service.kafka.servlet.KafkaCheckRunnable;
import io.transwarp.service.process.servlet.ProcessCheckRunnable;

public class ServiceCheck implements CheckInterface {
	
	private final ClusterInfo clusterInfo;
	private final List<String> hasCheckServiceNames;

	public ServiceCheck(final ClusterInfo clusterInfo) {
		this.clusterInfo = clusterInfo;
		this.hasCheckServiceNames = new ArrayList<String>();
	}
	
	@Override
	public void check() throws Exception {
		for (Entry<String, ServiceBean> entry : clusterInfo.getServices().entrySet()) {
			ServiceBean service = entry.getValue();
			addConfigCheck(service);
			if (GlobalArgs.checkSelect.isProcessCheck()) {
				addProcessCheck(service);
			}
			addLogCheck(service);
			if (GlobalArgs.checkSelect.isServiceCheckKafka() && service.getType().matches("\\S*KAFKA\\S*")) {
				String dependencies = service.getDependencies();
				String[] dependencyId = dependencies.substring(1, dependencies.length() - 1).split(",");
				ServiceBean zookeeper = getDependencyService(dependencyId, "ZOOKEEPER");
				if (zookeeper != null) {
					String zookeeperName = zookeeper.getName();
					if (!hasCheckServiceNames.contains(zookeeperName)) {
						hasCheckServiceNames.add(zookeeperName);
						Map<String, KafkaTopicBean> topicInfo = new ConcurrentHashMap<String, KafkaTopicBean>();
						MyThreadPool.addNewThread(new KafkaCheckRunnable(zookeeper, topicInfo));
						clusterInfo.addKafkaTopicBean(service.getName(), topicInfo);
					}
				}
			}
		}
	}

	private void addConfigCheck(final ServiceBean service) {
		String configFiles = clusterInfo.prop_config_name.getProperty(service.getType(), "");
		if (configFiles.equals("")) {
			return;
		}
		MyThreadPool.addNewThread(new ConfigCheckRunnable(service, configFiles.split(","), clusterInfo.prop_service_check));	
	}
	
	private void addProcessCheck(final ServiceBean service) {
		String servicename = service.getName();
		List<RoleBean> roles = service.getRoles();
		for (RoleBean role : roles) {
			if (role.getRoleType().equals("DOWN")) {
				continue;
			}
			String checkIdCommand = clusterInfo.prop_process_check.getProperty("process." + role.getRoleType().toLowerCase(Locale.getDefault()) + ".id", "");
			if (!checkIdCommand.equals("")) {
				ProcessBean process = new ProcessBean();
				process.setServicename(servicename);
				process.setRoleType(role.getRoleType());
				service.addProcessBean(process);
				MyThreadPool.addNewThread(new ProcessCheckRunnable(process, role.getNode().getIpAddress(), clusterInfo.prop_process_check));
			}
			
		}
	}
	
	private void addLogCheck(final ServiceBean service) {
		
	}	
	
	private ServiceBean getDependencyService(final String[] ids, final String match) {
		for (Entry<String, ServiceBean> entry : clusterInfo.getServices().entrySet()) {
			ServiceBean service = entry.getValue();
			String id = service.getId();
			String serviceType = service.getType();
			if (serviceType.matches("\\S*" + match + "\\S*")) {
				for (String dependencyId : ids) {
					if (dependencyId.equals(id)) {
						return service;
					}
				}
			}
		}
		return null;
	}

}
