package io.transwarp.table.comparison;

import io.transwarp.common.GlobalArgs;
import io.transwarp.common.bean.table.EsTableBean;
import io.transwarp.common.util.UtilTools;
import io.transwarp.report.comparison.Comparison;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import net.sf.json.JSONObject;

public class ESTableComparisonImpl implements Comparison {
	
	private final ComparisonResultOfTable comparisonResult;
	private final String servicename;
	private final Map<String, EsTableBean> tables;
	private Long shardSizeLimit;
	
	private final Map<String, Long> totalSizes;
	
	public ESTableComparisonImpl(final ComparisonResultOfTable comparisonResult,
			final String servicename,
			final Map<String, EsTableBean> tables) {
		this.comparisonResult = comparisonResult;
		this.servicename = servicename;
		this.tables = tables;
		this.totalSizes = new HashMap<String, Long>();
		this.shardSizeLimit = Long.valueOf(GlobalArgs.prop_rules.getProperty("table.es.shard.size.limit", "25"))*1024*1024*1024;
	}
	
	@Override
	public void comparison() {
		List<EsTableBean> trueTables = new ArrayList<EsTableBean>();
		List<EsTableBean> errorTables = new ArrayList<EsTableBean>();
		for (Entry<String, EsTableBean> entry : tables.entrySet()) {
			EsTableBean table = entry.getValue();
			Long maxSize = 0L;
			JSONObject checkResult = checkShard(table.getPrimaryShard()); 
			maxSize = Math.max(maxSize, checkResult.getLong("maxSize"));
			table.addShardComparison("primary", checkResult);
			List<Map<String, JSONObject>> replicaShard = table.getReplicaShardLists();
			for (Map<String, JSONObject> replicaInfo : replicaShard) {
				JSONObject result = checkShard(replicaInfo);
				maxSize = Math.max(maxSize, result.getLong("maxSize"));
				table.addShardComparison("replica", result);
			}
			if (maxSize > shardSizeLimit) {
				table.setWarning("shard over size");
				errorTables.add(table);
				comparisonResult.esErrorInfos.add(new String[]{String.format("%s.%s.%s.%s", servicename, table.getDatabase(), table.getTablename(), table.getEsTablename()),
						"shard over size",
						String.format("max shard size is %s", UtilTools.getCarrySize(maxSize))});
			} else {
				table.setWarning("null");
				trueTables.add(table);
			}
		}
		comparisonResult.esTrueTables.put(servicename, sortTables(trueTables));
		comparisonResult.esErrorTables.put(servicename, sortTables(errorTables));
		comparisonResult.searchServices.put(servicename, totalSizes);
	}
	
	private EsTableBean[] sortTables(List<EsTableBean> tables) {
		int number = tables.size();
		EsTableBean[] orders = new EsTableBean[number];
		return tables.toArray(orders);
	}
	
	private JSONObject checkShard(final Map<String, JSONObject> shardInfo) {
		Map<String, List<Long>> allShardSizes = new HashMap<String, List<Long>>();
		Long totalSize = 0L;
		int totalNumber = 0;
		Long maxSize = 0L;
		for (Entry<String, JSONObject> entry : shardInfo.entrySet()) {
			JSONObject json = entry.getValue();
			String hostname = json.getString("host");
			List<Long> shardSizes = allShardSizes.get(hostname);
			if (shardSizes == null) {
				shardSizes = new ArrayList<Long>();
				allShardSizes.put(hostname, shardSizes);
			}
			Long size = json.getLong("size");
			Long value = totalSizes.get(hostname);
			if (value == null) {
				totalSizes.put(hostname, size);
			} else {
				totalSizes.put(hostname, size + value);
			}
			maxSize = Math.max(maxSize, size);
			shardSizes.add(size);
			totalSize += size;
			totalNumber += 1;
		}
		JSONObject shardSizeInfo = new JSONObject();
		shardSizeInfo.put("totalSize", UtilTools.getCarrySize(totalSize));
		shardSizeInfo.put("totalNumber", totalNumber);
		shardSizeInfo.put("maxSize", maxSize);
		for (Entry<String, List<Long>> entry : allShardSizes.entrySet()) {
			String hostname = entry.getKey();
			List<Long> shardSizes = entry.getValue();
			int number = shardSizes.size();
			if (number == 0) {
				shardSizeInfo.put(hostname, " | | | ");
				continue;
			}
			Long[] sizes = new Long[number];
			sizes = shardSizes.toArray(sizes);
			Arrays.sort(sizes);
			shardSizeInfo.put(hostname, String.format("%s|%s|%s|%s", UtilTools.getCarrySize(sizes[number - 1]), 
					UtilTools.getCarrySize(sizes[number/2]), UtilTools.getCarrySize(sizes[0]), number));
		}
		return shardSizeInfo;
	}
}
