package io.transwarp.table.comparison;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import io.transwarp.common.GlobalArgs;
import io.transwarp.common.bean.table.HBaseTableBean;
import io.transwarp.common.util.UtilTools;
import io.transwarp.report.comparison.Comparison;

public class HBaseComparisonImpl implements Comparison {
	
	private final ComparisonResultOfTable comparisonResult;
	private final String servicename;
	private final Map<String, HBaseTableBean> tables;
	private long regionSizeLimit;
	
	private final Map<String, Integer> regionNumber;
	

	public HBaseComparisonImpl(final ComparisonResultOfTable comparisonResult,
			final String servicename,
			final Map<String, HBaseTableBean> tables) {
		this.comparisonResult = comparisonResult;
		this.servicename = servicename;
		this.tables = tables;
		this.regionNumber = new HashMap<String, Integer>();
		this.regionSizeLimit = Long.valueOf(GlobalArgs.prop_rules.getProperty("table.hbase.region.size.limit", "10"))*1024*1024*1024;
	}
	
	@Override
	public void comparison() {	
		List<HBaseTableBean> trueTables = new ArrayList<HBaseTableBean>();
		List<HBaseTableBean> errorTables = new ArrayList<HBaseTableBean>();
		for (Entry<String, HBaseTableBean> entry : tables.entrySet()) {
			HBaseTableBean table = entry.getValue();
			String warning = check(table);
			table.setWarning(warning);
			if (warning.equals("null")) {
				trueTables.add(table);
			} else {
				errorTables.add(table);
			}
		}
		comparisonResult.hbaseTrueTables.put(servicename, sortTables(trueTables));
		comparisonResult.hbaseErrorTables.put(servicename, sortTables(errorTables));
		comparisonResult.regionNumber.put(servicename, regionNumber);
	}
	
	private HBaseTableBean[] sortTables(List<HBaseTableBean> tables) {
		int number = tables.size();
		if (number == 0) {
			return null;
		}
		HBaseTableBean[] orderTables = new HBaseTableBean[number];
		orderTables = tables.toArray(orderTables);
		Arrays.sort(orderTables, new Comparator<HBaseTableBean>() {
			@Override
			public int compare(HBaseTableBean table1, HBaseTableBean table2) {
				int total1 = table1.getTotalRegionNumber();
				int total2 = table2.getTotalRegionNumber();
				if (total1 > total2) {
					return -1;
				} else if (total1 == total2) {
					return 0;
				} else {
					return 1;
				}
			}
		});
		return orderTables;
	}
	
	private String check(final HBaseTableBean table) {
		Map<String, List<String>> regionDistribution = table.getRegionDistribution();
		Map<String, Long> regionSizes = table.getRegionSizes();
		Long maxSize = 0L;
		for (Entry<String, List<String>> entry : regionDistribution.entrySet()) {
			String hostname = entry.getKey();
			List<String> regionNames = entry.getValue();
			int regionNumber = regionNames.size();
			addRegionNumberByHost(hostname, regionNumber);
			Long[] orderRegionSize = new Long[regionNumber];
			int index = 0;
			for (String regionName : regionNames) {
				orderRegionSize[index] = regionSizes.get(regionName);
				maxSize = Math.max(maxSize, orderRegionSize[index]);
				index += 1;
			}
			Arrays.sort(orderRegionSize);
			table.addRegionComparison(hostname, String.format("%s|%s|%s|%s", UtilTools.getCarrySize(orderRegionSize[regionNumber - 1]),
					UtilTools.getCarrySize(orderRegionSize[regionNumber/2]),
					UtilTools.getCarrySize(orderRegionSize[0]),
					regionNumber));
		}
		if (maxSize > regionSizeLimit) {
			comparisonResult.hbaseErrorInfos.add(new String[]{String.format("%s.%s.%s.%s", servicename, table.getDatabase(), table.getTablename(), table.getHbaseTablename()),
					"region over size",
					String.format("max region size is %s", UtilTools.getCarrySize(maxSize))});
			return "region over size";
		} else {
			return "null";
		}
	}
	
	private void addRegionNumberByHost(final String hostname, final int regionNumber) {
		Integer hasRegionNumber = this.regionNumber.get(hostname);
		if (hasRegionNumber == null) {
			hasRegionNumber = 0;
		}
		hasRegionNumber += regionNumber;
		this.regionNumber.put(hostname, hasRegionNumber);
	}
}
