package io.transwarp.table.hbase.report;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import jxl.write.Label;
import jxl.write.WritableCellFormat;
import jxl.write.WritableSheet;
import jxl.write.WritableWorkbook;
import io.transwarp.common.bean.TableInfo;
import io.transwarp.common.bean.table.HBaseTableBean;
import io.transwarp.common.util.UtilTools;
import io.transwarp.report.xls.ReportEnum;
import io.transwarp.report.xls.XlsReport;
import io.transwarp.table.comparison.ComparisonResultOfTable;

public class HBaseXlsReportImpl implements XlsReport {

	private final WritableSheet sheet;
	private final TableInfo tableInfo;
	private final ComparisonResultOfTable comparisonResult;
	private final Map<String, Integer> totalRegionCount;
	private final List<String> regionServerOrder;
	private boolean encrypt;
	private int tableId;
	private int maxCol;
	
	public HBaseXlsReportImpl(final WritableWorkbook workbook,
			final ReportEnum checkItem,
			final TableInfo tableInfo,
			final ComparisonResultOfTable comparisonResult) {
		this.sheet = workbook.createSheet("ddl_hbase", checkItem.ordinal());
		this.tableInfo = tableInfo;
		this.comparisonResult = comparisonResult;
		this.totalRegionCount = new HashMap<String, Integer>();
		this.regionServerOrder = new ArrayList<String>();
		this.encrypt = tableInfo.isEncryptTablename();
		this.tableId = 1;
		this.maxCol = 1;
	}
	
	@Override
	public void writeSheet() throws Exception {
		int beginRow = 3;
		int totalTable = 0;
		for (Entry<String, Map<String, HBaseTableBean>> entry : tableInfo.TABLE_HBASE.getAllHBaseTables().entrySet()) {
			String servicename = entry.getKey();
			Map<String, HBaseTableBean> tables = entry.getValue();;
			if (tables.size() == 0) {
				continue;
			}
			totalTable += tables.size();
			sheet.addCell(new Label(0, beginRow, servicename, CELL_BOLD));
			sheet.addCell(new Label(1, beginRow, "number of hbase table : " + tables.size(), CELL_THIN));
			beginRow += 1;
			beginRow = writeHBaseServiceInfo(beginRow, servicename);
			beginRow += 1;
		}
		if (totalTable == 0) {
			sheet.addCell(new Label(1, 0, "no find hbase table", CELL_THIN));
		} else {
			sheet.addCell(new Label(1, 0, "hostname", CELL_BOLD));
			sheet.addCell(new Label(1, 1, "total region number", CELL_BOLD));
			int column = 2;
			for (Entry<String, Integer> entry : this.totalRegionCount.entrySet()) {
				sheet.addCell(new Label(column, 0, entry.getKey(), CELL_THIN));
				sheet.addCell(new Label(column, 1, String.valueOf(entry.getValue()), CELL_THIN));
				column += 1;
			}
		}
		for (int i = 0; i < maxCol; i++) {
			sheet.setColumnView(i, 20);
		}
	}
	
	
	private int writeHBaseServiceInfo(int beginRow, final String servicename) throws Exception {
		Map<String, Integer> regionCount = comparisonResult.regionNumber.get(servicename);
		beginRow = writeRegionInTransition(beginRow, servicename);
		beginRow = writeRegionNumberCount(beginRow, regionCount);
		beginRow = writeHBaseParameter(beginRow, servicename);
		beginRow = writeAllTableInfos(beginRow, comparisonResult.hbaseTrueTables.get(servicename), 
				comparisonResult.hbaseErrorTables.get(servicename), regionCount);
		return beginRow;
	}

	private int writeRegionInTransition(int beginRow, final String servicename) throws Exception {
		Map<String, String> regionInTransitions = tableInfo.TABLE_HBASE.getRegionInTransitions(servicename);
		if (regionInTransitions.size() == 0) {
			return beginRow;
		}
		sheet.addCell(new Label(1, beginRow++, "region in transition", CELL_BOLD));
		sheet.mergeCells(2, beginRow, 4, beginRow);
		sheet.addCell(new Label(2, beginRow, "key", CELL_BOLD));
		sheet.mergeCells(5, beginRow, 7, beginRow);
		sheet.addCell(new Label(5, beginRow, "value", CELL_BOLD));
		beginRow += 1;
		for (Entry<String, String> entry : regionInTransitions.entrySet()) {
			String key = entry.getKey();
			String value = entry.getValue();
			sheet.mergeCells(2, beginRow, 4, beginRow);
			sheet.addCell(new Label(2, beginRow, key, CELL_THIN));
			sheet.mergeCells(5, beginRow, 7, beginRow);
			sheet.addCell(new Label(5, beginRow, value, CELL_THIN));
			beginRow += 1;
		}
		return beginRow;
	}
	
	private int writeRegionNumberCount(int beginRow, Map<String, Integer> regionCount) throws Exception {
		if (regionCount.size() == 0) {
			return beginRow;
		}
		int column = 1;
		sheet.addCell(new Label(column++, beginRow++, "region number count", CELL_BOLD));
		sheet.addCell(new Label(column, beginRow, "hostname", CELL_BOLD));
		sheet.addCell(new Label(column, beginRow + 1, "region number", CELL_BOLD));
		for (Entry<String, Integer> entry : regionCount.entrySet()) {
			String hostname = entry.getKey();
			Integer number = entry.getValue();
			addRegionNumber(hostname, number);
			sheet.addCell(new Label(column, beginRow, hostname, CELL_THIN));
			sheet.addCell(new Label(column, beginRow + 1, String.valueOf(number), CELL_THIN));
			column += 1;
		}
		return beginRow + 2;
	}
	
	private int writeHBaseParameter(int beginRow, final String servicename) throws Exception {
		Map<String, String> parameters = tableInfo.TABLE_HBASE.getRegionParameter(servicename);
		int column = 1;
		sheet.addCell(new Label(column++, beginRow++, "hbase.hregion.max.filesize", CELL_BOLD));
		sheet.addCell(new Label(column, beginRow, "hostname", CELL_BOLD));
		sheet.addCell(new Label(column, beginRow + 1, "parameter value", CELL_BOLD));
		for (Entry<String, String> entry : parameters.entrySet()) {
			String ipAddress = entry.getKey();
			String value = entry.getValue();
			sheet.addCell(new Label(column, beginRow, ipAddress, CELL_THIN));
			sheet.addCell(new Label(column, beginRow + 1, UtilTools.getCarrySize(value), CELL_THIN));
			column += 1;
		}
		return beginRow + 2;
	}
	
	private int writeAllTableInfos(int beginRow, 
			final HBaseTableBean[] trueTables, 
			final HBaseTableBean[] errorTables,
			final Map<String, Integer> regionCount) throws Exception {
		int column = 1;
		sheet.addCell(new Label(column++, beginRow++, "table information", CELL_BOLD));
		beginRow = writeTitle(column, beginRow, regionCount);
		if (errorTables != null) {
			for (HBaseTableBean table : errorTables) {
				writeTableInfo(beginRow, column, table, CELL_ERROR);
				beginRow += 1;
			}
		}
		if (trueTables != null) {
			for (HBaseTableBean table : trueTables) {
				writeTableInfo(beginRow, column, table, CELL_THIN);
				beginRow += 1;
			}
		}
		return beginRow;
	}
	
	private void addRegionNumber(final String hostname, final Integer number) {
		Integer totalNumber = this.totalRegionCount.get(hostname);
		if (totalNumber == null) {
			totalNumber = 0;
		}
		totalNumber += number;
		this.totalRegionCount.put(hostname, totalNumber);
	}
	
	private int writeTitle(int column, int beginRow, Map<String, Integer> regionCount) throws Exception {
		sheet.mergeCells(column, beginRow, column, beginRow + 2);
		sheet.addCell(new Label(column++, beginRow, "warning", CELL_BOLD));
		if (encrypt) {
			sheet.mergeCells(column, beginRow, column, beginRow + 2);
			sheet.addCell(new Label(column++, beginRow, "table id", CELL_BOLD));
		} else {
			sheet.mergeCells(column, beginRow, column, beginRow + 2);
			sheet.addCell(new Label(column++, beginRow, "database name", CELL_BOLD));
			sheet.mergeCells(column, beginRow, column, beginRow + 2);
			sheet.addCell(new Label(column++, beginRow, "table name", CELL_BOLD));
			sheet.mergeCells(column, beginRow, column, beginRow + 2);
			sheet.addCell(new Label(column++, beginRow, "hbase table", CELL_BOLD));
		}
		sheet.mergeCells(column, beginRow, column, beginRow + 2);
		sheet.addCell(new Label(column++, beginRow, "total size", CELL_BOLD));
		sheet.mergeCells(column, beginRow, column, beginRow + 2);
		sheet.addCell(new Label(column++, beginRow, "total region number", CELL_BOLD));
		if (regionCount.size() > 0) {
			int beginCol = column;
			for (Entry<String, Integer> entry : regionCount.entrySet()) {
				String hostname = entry.getKey();
				this.regionServerOrder.add(hostname);
				sheet.mergeCells(column, beginRow + 1, column + 3, beginRow + 1);
				sheet.addCell(new Label(column, beginRow + 1, String.format("%s[%s]", hostname, tableInfo.TABLE_HBASE.getRegionStartTime(hostname)), CELL_BOLD));
				sheet.addCell(new Label(column++, beginRow + 2, "max size", CELL_BOLD));
				sheet.addCell(new Label(column++, beginRow + 2, "median size", CELL_BOLD));
				sheet.addCell(new Label(column++, beginRow + 2, "min size", CELL_BOLD));
				sheet.addCell(new Label(column++, beginRow + 2, "region number", CELL_BOLD));
			}
			sheet.mergeCells(beginCol, beginRow, column - 1, beginRow);
			sheet.addCell(new Label(beginCol, beginRow, "region size", CELL_BOLD));
		}
		this.maxCol = Math.max(maxCol, column);
		return beginRow + 3;
	}
	
	private void writeTableInfo(int beginRow, int beginCol, final HBaseTableBean table, final WritableCellFormat cell) throws Exception {
		sheet.addCell(new Label(beginCol++, beginRow, table.getWarning(), cell));
		if (encrypt) {
			tableInfo.TABLE_HBASE.addEncryptHBase(tableId, String.format("%s.%s.%s", table.getDatabase(), table.getTablename(), table.getHbaseTablename()));
			sheet.addCell(new Label(beginCol++, beginRow, String.valueOf(tableId++), cell));
		} else {
			sheet.addCell(new Label(beginCol++, beginRow, table.getDatabase(), cell));
			sheet.addCell(new Label(beginCol++, beginRow, table.getTablename(), cell));
			sheet.addCell(new Label(beginCol++, beginRow, table.getHbaseTablename(), cell));
		}
		sheet.addCell(new Label(beginCol++, beginRow, UtilTools.getCarrySize(table.getTotalRegionSize()), cell));
		sheet.addCell(new Label(beginCol++, beginRow, String.valueOf(table.getTotalRegionNumber()), cell));
		Map<String, String> comparisons = table.getRegionComparison();
		for (String hostname : this.regionServerOrder) {
			String comparison = comparisons.get(hostname);
			if (comparison == null) {
				beginCol += 4;
				continue;
			}
			int column = beginCol;
			String[] items = comparison.split("\\|");
			for (String item : items) {
				sheet.addCell(new Label(column++, beginRow, item, cell));
			}
			beginCol += 4;
		}
	}
	
}
