package io.transwarp.table.hbase.servlet;

import java.text.SimpleDateFormat;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.hadoop.hbase.ClusterStatus;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.client.HBaseAdmin;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.master.RegionState;
import org.apache.log4j.Logger;

import io.transwarp.common.bean.TableInfo;
import io.transwarp.common.bean.service.ConfigBean;
import io.transwarp.common.bean.service.ServiceBean;
import io.transwarp.common.bean.table.HBaseTableBean;
import io.transwarp.common.util.MyThreadPool;
import io.transwarp.table.servlet.AbstractCheckOnDetail;

public class HBaseTableCheckOnDetail extends AbstractCheckOnDetail {
	
	private final Logger log = Logger.getLogger(HBaseTableCheckOnDetail.class);
    private static final long MAX_IN_TRANSITION_TIME = 1 * 60 * 1000;
	
	private final ServiceBean hyperbase;
	private final Map<String, HBaseTableBean> hbaseTables;
	private final TableInfo tableInfo;
	private final String hdfsName;
	
	private HyperbaseUtils method;
	
	public HBaseTableCheckOnDetail(final String hdfsName,
			final ServiceBean hyperbase,
			final Map<String, HBaseTableBean> tables,
			final TableInfo tableInfo) {
		this.hdfsName = hdfsName;
		this.hyperbase = hyperbase;
		this.hbaseTables = tables;
		this.tableInfo = tableInfo;
	}

	@Override
	public void addTableCheck() throws Exception {
		if (hyperbase == null || hyperbase.getStatus().equals("DOWN")) {
			return;
		}
		this.method = HyperbaseUtils.custom().setService(hyperbase).build();
		if (method == null) {
			throw new Exception("build hyperbase connected faild");
		}
		String sid = hyperbase.getSid();
		HBaseAdmin admin = null;
		try {
			admin = method.getAdmin();
			ClusterStatus clusterStatus = admin.getClusterStatus();
			getRegionInTransitions(clusterStatus);
			HTableDescriptor[] tableDescriptors = admin.listTables();
			for (HTableDescriptor tableDescriptor : tableDescriptors) {
				try {
					getHBaseTableInfo(sid, tableDescriptor);
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
			getRegionParameter();
			addHBaseTableCheck();
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (admin != null) {
				try {
					admin.close();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
	}
	
	private void getRegionInTransitions(ClusterStatus clusterStatus) {
		Map<String, String> regionInTransition = new HashMap<String, String>();
		Map<String, RegionState> rits = clusterStatus.getRegionsInTransition();
		for (Entry<String, RegionState> entry : rits.entrySet()) {
			RegionState regionState = entry.getValue();
			if (isBadState(regionState)) {
				regionInTransition.put(entry.getKey(), regionState.toDescriptiveString());
			}
		}
		tableInfo.TABLE_HBASE.addRegionInTransitions(hyperbase.getName(), regionInTransition);
	}
	
	private Boolean isBadState(RegionState state) {
		boolean isBadState = false;
        switch (state.getState()) {
            case PENDING_OPEN:
            case OPENING:
            case PENDING_CLOSE:
            case CLOSING:
            case SPLITTING:
            case MERGING:
            case SPLITTING_NEW:
            case MERGING_NEW:
                long realTime = System.currentTimeMillis() - state.getStamp();
                if (realTime > MAX_IN_TRANSITION_TIME) {
                    log.info("Region is in bad state: " + state.toDescriptiveString());
                    isBadState = true;
                }
                break;
            case FAILED_OPEN:
            case FAILED_CLOSE:
                log.info("Region is in bad state: " + state.toDescriptiveString());
                isBadState = true;
                break;
            case OFFLINE:
            case OPEN:
            case CLOSED:
            case SPLIT:
            case MERGED:
            default:
                isBadState = false;
                break;
        }
        return isBadState;
	}

	private void getHBaseTableInfo(String sid, HTableDescriptor tableDescriptor) throws Exception {
		String hbasename = tableDescriptor.getNameAsString();
		HBaseTableBean table = getHBaseTable(hbasename);
		String[] splitOfName = hbasename.split("\\:");
		if (splitOfName.length == 1) {
			table.setLocation(String.format("/%s/data/default/%s", sid, hbasename));
		} else {
			table.setLocation(String.format("/%s/data/%s/%s", sid, splitOfName[0], splitOfName[1]));
		}
		
		HColumnDescriptor[] families = tableDescriptor.getColumnFamilies();
		for (HColumnDescriptor family : families) {
			table.addFamily(family.getNameAsString());
		}
		SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		HTable htable = method.getHTable(hbasename);
		Map<HRegionInfo, ServerName> regionInfos = htable.getRegionLocations();
		for (Entry<HRegionInfo, ServerName> entry : regionInfos.entrySet()) {
			HRegionInfo regionKey = entry.getKey();
			ServerName server = entry.getValue();
			String hostname = server.getHostname();
			table.addRegionDistribution(hostname, regionKey.getEncodedName());
			tableInfo.TABLE_HBASE.addRegionStartTime(hostname, dateFormat.format(server.getStartcode()));
		}
	}
	
	private HBaseTableBean getHBaseTable(String tablename) {
		for (Entry<String, HBaseTableBean> entry : hbaseTables.entrySet()) {
			HBaseTableBean table = entry.getValue();
			if (table.getHbaseTablename().equals(tablename)) {
				return table;
			}
		}
		HBaseTableBean table = new HBaseTableBean();
		table.setDatabase("");
		table.setTablename("");
		table.setHBaseTablename(tablename);
		hbaseTables.put("InceptorTableNotExists:" + tablename, table);
		return table;
	}
	
	private void getRegionParameter() {
		ConfigBean config = hyperbase.getConfig();
		Map<String, String> parameter = new HashMap<String, String>();
		Map<String, Map<String, String>> configFiles = config.getConfigFiles("hbase-site.xml");
		for (Entry<String, Map<String, String>> entry : configFiles.entrySet()) {
			String ipAddress = entry.getKey();
			Map<String, String> fileValue = entry.getValue();
			parameter.put(ipAddress, fileValue.get("hbase.hregion.max.filesize"));
		}
		tableInfo.TABLE_HBASE.addRegionParameter(hyperbase.getName(), parameter);
	}
	
	private void addHBaseTableCheck() {
		for (Entry<String, HBaseTableBean> entry : hbaseTables.entrySet()) {
			HBaseTableBean table = entry.getValue();
			MyThreadPool.addNewThread(new HBaseTableCheckRunnable(hdfsName, table));
		}
	}
}
