package io.transwarp.table.hbase.servlet;

import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;

import io.transwarp.common.GlobalArgs;
import io.transwarp.common.bean.table.HBaseTableBean;
import io.transwarp.common.util.AbstractRunnable;
import io.transwarp.common.util.HdfsConnUtils;
import io.transwarp.inspection.CheckItemEnum;

public class HBaseTableCheckRunnable extends AbstractRunnable {
	
	private final HBaseTableBean table;
	private final HdfsConnUtils hdfsPool;
	private FileSystem fs;
	
	public HBaseTableCheckRunnable(final String hdfsName,
			final HBaseTableBean table) {
		this.hdfsPool = GlobalArgs.hdfsPools.get(hdfsName);
		this.table = table;
	}
	
	@Override
	public void startCheck() throws Exception {
		fs = hdfsPool.getHdfsConn();
		Map<String, List<String>> regionDistributions = table.getRegionDistribution();
		List<String> families = table.getFamilies();
		String location = table.getLocation();
		for (Entry<String, List<String>> entry : regionDistributions.entrySet()) {
			List<String> regionList = entry.getValue();
			for (String regionName : regionList) {
				Long regionSize = getRegionSize(location + "/" + regionName, families);
				table.addRegion(regionName, regionSize);
			}
		}
	}
	
	@Override
	public void dealWithError(Exception e) {
		GlobalArgs.ERROR_INFO.add(String.format("check hbase table [%s.%s.%s] faild | %s", table.getDatabase(), table.getTablename(), table.getHbaseTablename(), e.getMessage()));
		e.printStackTrace();		
	}
	
	@Override
	public void closeThread() {
		hdfsPool.closeHdfsConn(fs);
		GlobalArgs.timeStatisticses.get(CheckItemEnum.TABLE.name()).setEndTime();
	}

	private Long getRegionSize(String path, List<String> families) {
		Long size = 0L;
		for (String family : families) {
			Long familySize = deepSearch(path + "/" + family);
			size += familySize;
		}
		return size;
	}
	
	private Long deepSearch(String path) {
		Long totalSize = 0L;
		try {
			FileStatus[] files = fs.listStatus(new Path(path));
			for (FileStatus file : files) {
				if (file.isDirectory()) {
					totalSize += deepSearch(file.getPath().toString());
				} else {
					totalSize += file.getLen();
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return totalSize;
	}
}
