package io.transwarp.table.hbase.servlet;

import java.io.IOException;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;

import io.transwarp.common.GlobalArgs;
import io.transwarp.common.bean.node.NodeBean;
import io.transwarp.common.bean.service.ConfigBean;
import io.transwarp.common.bean.service.RoleBean;
import io.transwarp.common.bean.service.ServiceBean;
import io.transwarp.common.util.ExecShell;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.client.HBaseAdmin;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.security.UserGroupInformation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HyperbaseUtils {

	private final static Logger log = LoggerFactory.getLogger(HyperbaseUtils.class);
	private final Configuration config;
	
	private HyperbaseUtils(Configuration config) {
		this.config = config;
	}
	
	public HBaseAdmin getAdmin() {
		HBaseAdmin admin = null;
		try {
			admin = new HBaseAdmin(config);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return admin;
	}
	
	public HTable getHTable(final String tablename) {
		HTable table = null;
		try {
			table = new HTable(config, tablename);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return table;
	}
	
	public static HyperbaseUtils.Build custom() {
		return new Build();
	}
	
	public static class Build {
		private ServiceBean service;
		
		Build() {
			service = null;
		}
		
		public HyperbaseUtils build() {
			if (service == null) {
				return null;
			}
			String sid = service.getSid();
			Boolean openKerberos = service.getEnableKerberos().equals("true") ? true : false;
			ConfigBean serviceConfig = null;
			int count = 0;
			while (serviceConfig == null) {
				serviceConfig = service.getConfig();
				count += 1;
				if (count > 60) {
					log.error("no find hyperbase config of service [{}]");
					break;
				}
				try {
					Thread.sleep(1000);
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
			List<RoleBean> roles = service.getRoles();
			for (RoleBean role : roles) {
				String type = role.getRoleType();
				if (type.matches("\\S*MASTER\\S*")) {
					return buildHyperbaseUtil(openKerberos, sid, role, serviceConfig);
				}
			}
			return null;
		}
		
		private HyperbaseUtils buildHyperbaseUtil(Boolean openKerberos, String sid, RoleBean role, ConfigBean serviceConfig) {
			NodeBean node = role.getNode();
			String ipAddress = node.getIpAddress();
			String hostname = node.getHostName().toLowerCase(Locale.getDefault());
			Map<String, String> fileValue = serviceConfig.getConfigFile(ipAddress, "hbase-site.xml");
			Configuration config = HBaseConfiguration.create();
			for (Entry<String, String> entry : fileValue.entrySet()) {
				config.set(entry.getKey(), entry.getValue());
			}
			if (openKerberos) {
				String keyPath = null;
				if (GlobalArgs.version.startsWith("4")) {
					keyPath = ExecShell.getKeytab(ipAddress, sid, "hbase.keytab");
				} else {
					keyPath = ExecShell.getKeytab(ipAddress, sid, "hyperbase.keytab");
				}
				config.set("hadoop.security.authentication", "kerberos");
				UserGroupInformation.setConfiguration(config);
				try {
					UserGroupInformation.loginUserFromKeytab("hbase/" + hostname, keyPath);
				} catch (IOException e) {
					e.printStackTrace();
					return null;
				}
			}
			return new HyperbaseUtils(config);
		}
		
		public Build setService(final ServiceBean service) {
			this.service = service;
			return this;
		}
	}
}
