package io.transwarp.table.holodesk.report;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import jxl.write.Label;
import jxl.write.WritableSheet;
import jxl.write.WritableWorkbook;
import io.transwarp.common.bean.TableInfo;
import io.transwarp.common.bean.table.HolodeskTableBean;
import io.transwarp.common.util.UtilTools;
import io.transwarp.report.xls.ReportEnum;
import io.transwarp.report.xls.XlsReport;
import io.transwarp.table.comparison.ComparisonResultOfTable;

public class HolodeskXlsReportImpl implements XlsReport {
	
	private final WritableSheet sheet;
	private final TableInfo tableInfo;
	private final ComparisonResultOfTable comparisonResult;
	private final Map<String, Long> totalBlockSize;
	private final boolean encrypt;
	private int tableId;
	private int maxCol;
	
	public HolodeskXlsReportImpl(final WritableWorkbook workbook,
			final ReportEnum checkItem,
			final TableInfo tableInfo,
			final ComparisonResultOfTable comparisonResult) {
		this.sheet = workbook.createSheet("ddl_holodesk", checkItem.ordinal());
		this.tableInfo = tableInfo;
		this.comparisonResult = comparisonResult;
		this.totalBlockSize = new HashMap<String, Long>();
		this.encrypt = tableInfo.isEncryptTablename();
		this.tableId = 1;
		this.maxCol = 0;
	}

	@Override
	public void writeSheet() throws Exception {
		int beginRow = 3;
		int totalTable = 0;
		for (Entry<String, Map<String, HolodeskTableBean>> entry : tableInfo.TABLE_HOLODESK.getAllHolodeskTables().entrySet()) {
			String servicename = entry.getKey();
			Map<String, HolodeskTableBean> tables = entry.getValue();
			if (tables.size() == 0) {
				continue;
			}
			totalTable += tables.size();
			sheet.addCell(new Label(0, beginRow, servicename, CELL_BOLD));
			sheet.addCell(new Label(1, beginRow, "number of holodesk : " + tables.size(), CELL_THIN));
			beginRow += 1;
			beginRow = writeParameter(beginRow, servicename);
			beginRow = writeTableInService(beginRow, servicename);
			beginRow += 1;
		}
		if (totalTable == 0) {
			sheet.addCell(new Label(1, 0, "find holodesk table", CELL_THIN));
		} else {
			writeTotalBlockSizes();
		}
		setColumnSize();
	}
	
	private int writeParameter(final int beginRow, final String servicename) throws Exception {
		sheet.addCell(new Label(1, beginRow, "ngmr.fastdisk.dir", CELL_BOLD));
		sheet.addCell(new Label(1, beginRow + 1, "ngmr.fastdisk.size", CELL_BOLD));
		
		Map<String, String> parameter = tableInfo.TABLE_HOLODESK.getHolodeskParameter(servicename);
		String dir = parameter.get("ngmr.fastdisk.dir");
		sheet.addCell(new Label(2, beginRow, dir, CELL_THIN));
		String size = parameter.get("ngmr.fastdisk.size");
		sheet.addCell(new Label(2, beginRow + 1, size, CELL_THIN));
		return beginRow + 2;
	}
	
	private int writeTableInService(int beginRow, final String servicename) throws Exception {
		HolodeskTableBean[] holodesks = comparisonResult.holodeskTables.get(servicename);
		Set<String> hostnames = comparisonResult.holodeskHosts.get(servicename);
		beginRow = writeTitle(beginRow, hostnames);
		for (HolodeskTableBean table : holodesks) {
			int column = 1;
			if (encrypt) {
				tableInfo.TABLE_HOLODESK.addEncryptHolodesk(tableId, String.format("%s.%s", table.getDatabase(), table.getTablename()));
				sheet.addCell(new Label(column++, beginRow, String.valueOf(tableId++), CELL_THIN));
			} else {
				sheet.addCell(new Label(column++, beginRow, table.getDatabase(), CELL_THIN));
				sheet.addCell(new Label(column++, beginRow, table.getTablename(), CELL_THIN));
			}
			Map<String, Long> blockSizes = table.getBlockSizes();
			for (String hostname : hostnames) {
				Long size = blockSizes.get(hostname);
				if (size == null) {
					sheet.addCell(new Label(column++, beginRow, "", CELL_THIN));
				} else {
					Long totalSize = this.totalBlockSize.get(hostname);
					if (totalSize == null) {
						totalSize = 0L;
					}
					totalSize += size;
					this.totalBlockSize.put(hostname, totalSize);
					sheet.addCell(new Label(column++, beginRow, UtilTools.getCarrySize(size), CELL_THIN));
				}
			}
			sheet.addCell(new Label(column++, beginRow, UtilTools.getCarrySize(table.getTotalSize()), CELL_THIN));
			beginRow += 1;
		}
		return beginRow;
	}
	
	private void writeTotalBlockSizes() throws Exception {
		int column = 1;
		sheet.addCell(new Label(column, 0, "hostname", CELL_BOLD));
		sheet.addCell(new Label(column, 1, "total block size", CELL_BOLD));
		for (Entry<String, Long> entry : this.totalBlockSize.entrySet()) {
			column += 1;
			sheet.addCell(new Label(column, 0, entry.getKey(), CELL_THIN));
			sheet.addCell(new Label(column, 1, UtilTools.getCarrySize(entry.getValue()), CELL_THIN));
		}
	}
	
	private int writeTitle(final int beginRow, final Set<String> hostnames) throws Exception {
		int column = 1;
		if (encrypt) {
			sheet.mergeCells(column, beginRow, column, beginRow + 1);
			sheet.addCell(new Label(column++, beginRow, "table id", CELL_BOLD));
		} else {
			sheet.mergeCells(column, beginRow, column, beginRow + 1);
			sheet.addCell(new Label(column++, beginRow, "database name", CELL_BOLD));
			sheet.mergeCells(column, beginRow, column, beginRow + 1);
			sheet.addCell(new Label(column++, beginRow, "table name", CELL_BOLD));
		}

		int beginCol = column;
		for (String hostname : hostnames) {
			sheet.addCell(new Label(column++, beginRow + 1, hostname, CELL_BOLD));
		}
		sheet.addCell(new Label(column++, beginRow + 1, "total", CELL_BOLD));
		sheet.mergeCells(beginCol, beginRow, column - 1, beginRow);
		sheet.addCell(new Label(beginCol, beginRow, "block size", CELL_BOLD));
		this.maxCol = Math.max(maxCol, column);
		return beginRow + 2;
	}
	
	private void setColumnSize() {
		for (int i = 0; i < maxCol; i++) {
			sheet.setColumnView(i, 20);
		}
	}
}
