package io.transwarp.table.holodesk.servlet;

import it.unimi.dsi.fastutil.io.FastByteArrayInputStream;

import java.io.DataInputStream;
import java.util.ArrayList;
import java.util.List;

import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.retry.RetryNTimes;
import org.apache.zookeeper.KeeperException.NoNodeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZkConnUtils {

	private static final Logger log = LoggerFactory.getLogger(ZkConnUtils.class);
	private static ZkConnUtils zkConn;
	private static String host;
	private static final int TIMEOUT = 5000;
	private static final int RETRYTIME = 3;
	private static final int HOLODESK_MAGIC_NUMBER = 75124;
	private CuratorFramework zkClient;
	
	public static ZkConnUtils getZKConnection(String host) throws Exception {
		if (zkConn == null || ZkConnUtils.host == null) {
			zkConn = new ZkConnUtils(host, 2181);
			ZkConnUtils.host = host;
		} else if (!ZkConnUtils.host.equals(host)) {
			zkConn.closeClient();
			zkConn = new ZkConnUtils(host, 2181);
			ZkConnUtils.host = host;
		}
		return zkConn;
	}
	
	private ZkConnUtils(String host, int port) throws Exception {
		super();
		zkClient = CuratorFrameworkFactory.builder()
				.connectString(host + ":" + port).retryPolicy(new RetryNTimes(RETRYTIME, TIMEOUT))
				.connectionTimeoutMs(TIMEOUT).build();
		zkClient.start();
	}
	
	public Long getVersion(String path) throws Exception {
		long max = 0;
		try {
			List<String> versions = getFileList(path);
			for(String version : versions) {
				max = Math.max(max, Long.valueOf(version));
			}
		} catch (Exception e) {
		}
		return max;
	}

	public List<String> getFileList(String path) throws Exception {
		List<String> fileList = new ArrayList<String>();
		try {
			fileList = zkClient.getChildren().forPath(path);
		} catch (NoNodeException e) {
			log.warn("no find this path [{}] in zookeeper, this table has no data", path);
		}
		return fileList;
	}
	
	public Long getFileSize(String path) throws Exception {
		Long fileSize = new Long(0);
		byte[] data = zkClient.getData().forPath(path);
		DataInputStream di = new DataInputStream(new FastByteArrayInputStream(data));
		if (data.length >= 4 && di.readInt() == HOLODESK_MAGIC_NUMBER) {
			while (di.available() > 0) {
				di.readLong();
				di.readInt();
				di.readInt();
				fileSize += di.readInt();
				di.readInt();
				di.readLong();
			}
		} else {
			while (di.available() > 0) {
				di.readLong();
				di.readInt();
				di.readInt();
				fileSize += di.readInt();
				di.readInt();
			}
		}
		di.close();
		return fileSize;
	}

	public void closeClient() throws Exception {
		if (zkClient != null) {
			zkClient.close();
		}
	}
	
	public static void close() throws Exception {
		if (zkConn != null) {
			zkConn.closeClient();
		}
	}
}
