package io.transwarp.table.orc.report;

import java.util.Map;
import java.util.Map.Entry;

import jxl.write.Label;
import jxl.write.WritableCellFormat;
import jxl.write.WritableSheet;
import jxl.write.WritableWorkbook;
import io.transwarp.common.bean.TableInfo;
import io.transwarp.common.bean.table.OrcTableBean;
import io.transwarp.report.xls.ReportEnum;
import io.transwarp.report.xls.XlsReport;
import io.transwarp.table.comparison.ComparisonResultOfTable;

public class OrcXlsReportImpl implements XlsReport {
	
	private final WritableSheet sheet;
	private final TableInfo tableInfo;
	private final ComparisonResultOfTable comparisonResult;
	
	private final boolean encryptTablename;
	
	private int columnLen;
	private int tableId;
	
	public OrcXlsReportImpl(final WritableWorkbook workbook,
			final ReportEnum checkItem,
			final TableInfo tableInfo,
			final ComparisonResultOfTable comparisonResult) {
		this.sheet = workbook.createSheet("ddl_orc", checkItem.ordinal());
		this.tableInfo = tableInfo;
		this.comparisonResult = comparisonResult;
		this.encryptTablename = tableInfo.isEncryptTablename();
		this.tableId = 1;
	}

	@Override
	public void writeSheet() throws Exception {
		int beginRow = 0;
		int totalTable = 0;
		for (Entry<String, Map<String, OrcTableBean>> entry : tableInfo.TABLE_ORC.getAllOrcTables().entrySet()) {
			String servicename = entry.getKey();
			Map<String, OrcTableBean> tables = entry.getValue();
			if (tables.size() == 0) {
				continue;
			}
			totalTable += tables.size();
			sheet.addCell(new Label(0, beginRow, servicename, CELL_BOLD));
			sheet.addCell(new Label(1, beginRow++, "number of orc table : " + tables.size(), CELL_THIN));
			beginRow = writeTableTitle(beginRow);
			beginRow = writeTablesToSheet(beginRow, servicename);
			beginRow += 1;
		}
		if (totalTable == 0) {
			sheet.addCell(new Label(1, 0, "no find orc table", CELL_THIN));
		}
		setColumnSize();
	}
	
	private int writeTableTitle(int beginRow) throws Exception {
		int column = 1;
		sheet.mergeCells(column, beginRow, column, beginRow + 1);
		sheet.addCell(new Label(column++, beginRow, "warning", CELL_BOLD));
		
		if (encryptTablename) {
			sheet.mergeCells(column,  beginRow,  column, beginRow + 1);
			sheet.addCell(new Label(column++, beginRow, "table id", CELL_BOLD));
		} else {
			sheet.mergeCells(column, beginRow, column, beginRow + 1);
			sheet.addCell(new Label(column++, beginRow, "database name", CELL_BOLD));
			
			sheet.mergeCells(column, beginRow, column, beginRow + 1);
			sheet.addCell(new Label(column++, beginRow, "table name", CELL_BOLD));
		}
		
		sheet.mergeCells(column, beginRow, column, beginRow + 1);
		sheet.addCell(new Label(column++, beginRow, "owner name", CELL_BOLD));
		
		sheet.mergeCells(column, beginRow, column, beginRow + 1);
		sheet.addCell(new Label(column++, beginRow, "transactional", CELL_BOLD));
		
		sheet.mergeCells(column, beginRow, column + 2, beginRow);
		sheet.addCell(new Label(column, beginRow, "delta menu number on partition", CELL_BOLD));
		sheet.addCell(new Label(column++, beginRow + 1, "max number", CELL_BOLD));
		sheet.addCell(new Label(column++, beginRow + 1, "min number", CELL_BOLD));
		sheet.addCell(new Label(column++, beginRow + 1, "error number", CELL_BOLD));
		
		sheet.mergeCells(column, beginRow, column, beginRow + 1);
		sheet.addCell(new Label(column++, beginRow, "bucket column", CELL_BOLD));
		
		sheet.mergeCells(column, beginRow, column + 3, beginRow);
		sheet.addCell(new Label(column, beginRow, "bucket file size", CELL_BOLD));
		sheet.addCell(new Label(column++, beginRow + 1, "max size", CELL_BOLD));
		sheet.addCell(new Label(column++, beginRow + 1, "median size", CELL_BOLD));
		sheet.addCell(new Label(column++, beginRow + 1, "min size", CELL_BOLD));
		sheet.addCell(new Label(column++, beginRow + 1, "bucket file number", CELL_BOLD));
		
		sheet.mergeCells(column, beginRow, column + 3, beginRow);
		sheet.addCell(new Label(column, beginRow, "data file size", CELL_BOLD));
		sheet.addCell(new Label(column++, beginRow + 1, "max size", CELL_BOLD));
		sheet.addCell(new Label(column++, beginRow + 1, "median size", CELL_BOLD));
		sheet.addCell(new Label(column++, beginRow + 1, "min size", CELL_BOLD));
		sheet.addCell(new Label(column++, beginRow + 1, "data file number", CELL_BOLD));
		
		sheet.mergeCells(column, beginRow, column, beginRow + 1);
		sheet.addCell(new Label(column++, beginRow, "partition key", CELL_BOLD));
		
		sheet.mergeCells(column, beginRow, column + 3, beginRow);
		sheet.addCell(new Label(column, beginRow, "partition size", CELL_BOLD));
		sheet.addCell(new Label(column++, beginRow + 1, "max size", CELL_BOLD));
		sheet.addCell(new Label(column++, beginRow + 1, "median size", CELL_BOLD));
		sheet.addCell(new Label(column++, beginRow + 1, "min size", CELL_BOLD));
		sheet.addCell(new Label(column++, beginRow + 1, "partition number", CELL_BOLD));
		
		sheet.mergeCells(column, beginRow, column, beginRow + 1);
		sheet.addCell(new Label(column++, beginRow, "small file number", CELL_BOLD));
		
		sheet.mergeCells(column, beginRow, column, beginRow + 1);
		sheet.addCell(new Label(column++, beginRow, "small file proportion", CELL_BOLD));
		
		columnLen = column - 1;
		return beginRow + 2;
	}
	
	private int writeTablesToSheet(int beginRow, String servicename) throws Exception {
		// output error table
		OrcTableBean[] errorTables = comparisonResult.orcErrorTables.get(servicename);
		if (errorTables != null) {
			for (OrcTableBean table : errorTables) {
				writeTableInfo(beginRow++, table, CELL_ERROR);
			}
		}
		// output true table
		OrcTableBean[] trueTables = comparisonResult.orcTrueTables.get(servicename);
		if (trueTables != null) {
			for (OrcTableBean table : trueTables) {
				writeTableInfo(beginRow++, table, CELL_THIN);
			}
		}
		return beginRow;
	}
	
	private void writeTableInfo(final int row, final OrcTableBean table, WritableCellFormat cell) throws Exception {
		int column = 1;
		sheet.addCell(new Label(column++, row, table.getWarning(), cell));
		
		if (encryptTablename) {
			sheet.addCell(new Label(column++, row, String.valueOf(tableId), cell));
			tableInfo.TABLE_ORC.addEncryptOrc(tableId++, String.format("%s.%s", table.getDatabase(), table.getTablename()));
		} else {
			sheet.addCell(new Label(column++, row, table.getDatabase(), cell));
			sheet.addCell(new Label(column++, row, table.getTablename(), cell));
		}
		
		sheet.addCell(new Label(column++, row, table.getOwner(), cell));
		sheet.addCell(new Label(column++, row, String.valueOf(table.isTransactional()), cell));
		
		String[] deltaComparisons = table.getDeltaComparison().split("\\|");
		for (String deltaItem : deltaComparisons) {
			sheet.addCell(new Label(column++, row, deltaItem, cell));
		}
		
		String bucketColumn = table.getBucketColumn();
		String[] bucketComparisons = table.getBucketComparison().split("\\|");
		sheet.addCell(new Label(column++, row, bucketColumn, cell));
		
		if (bucketColumn.equals("")) {
			for (int i = 0; i < 4; i++) {
				sheet.addCell(new Label(column++, row, "", cell));
			}
			for (String bucketItem : bucketComparisons) {
				sheet.addCell(new Label(column++, row, bucketItem, cell));
			}
		} else {
			for (String bucketItem : bucketComparisons) {
				sheet.addCell(new Label(column++, row, bucketItem, cell));
			}
			for (int i = 0; i < 4; i++) {
				sheet.addCell(new Label(column++, row, "", cell));
			}
		}
		
		String partitionKey = table.getPartitionKey();
		sheet.addCell(new Label(column++, row, partitionKey, cell));
		if (!partitionKey.equals("")) {
			String[] partitionComparisons = table.getPartitionComparison().split("\\|");
			for (String partitionItem : partitionComparisons) {
				sheet.addCell(new Label(column++, row, partitionItem, cell));
			}
		} else {
			for (int i = 0; i < 4; i++) {
				sheet.addCell(new Label(column++, row, "", cell));
			}
		}
		
		sheet.addCell(new Label(column++, row, String.valueOf(table.getSmallFileNumber()), cell));
		sheet.addCell(new Label(column++, row, table.getSmallFilePercent(), cell));
	}
	
	private void setColumnSize() {
		for (int i = 0; i <= columnLen; i++) {
			sheet.setColumnView(i, 25);
		}
	}
}
