package io.transwarp.table.orc.servlet;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.util.Map;
import java.util.Map.Entry;

import net.sf.json.JSONObject;

import org.apache.log4j.Logger;

import io.transwarp.common.bean.table.OrcTableBean;
import io.transwarp.common.util.JDBCConnected;
import io.transwarp.common.util.MyThreadPool;
import io.transwarp.table.servlet.AbstractCheckOnDetail;

public class OrcTableCheckOnDetail extends AbstractCheckOnDetail {
	
	private final Logger log = Logger.getLogger(OrcTableCheckOnDetail.class);
	/** mysql数据库查询分桶字段及分桶数 */
	public static final String QUERY_BUCKETS = "SELECT DBS.NAME database_name, TBLS.tbl_name table_name, col.bucket_col_name bucket_column, sds.num_buckets bucket_number FROM TBLS JOIN DBS ON TBLS.db_id = DBS.db_id JOIN SDS sds ON TBLS.sd_id = sds.sd_id JOIN BUCKETING_COLS col ON col.sd_id = TBLS.sd_id;";
	/** mysql数据库查询单值分区数 */
	public static final String QUERY_PARTITIONS = "select part.part_name partition_name, partition_key.pkey_name partition_key, TBLS.tbl_name table_name, DBS.name database_name from PARTITION_KEY_VALS partition_value join PARTITIONS part on part.part_id = partition_value.part_id join TBLS on part.tbl_id = TBLS.tbl_id join DBS on TBLS.db_id = DBS.db_id join PARTITION_KEYS partition_key on partition_key.tbl_id = TBLS.tbl_id;";
	/** mysql数据库查询范围分区数 */
	public static final String QUERY_PARTITIONS_RANGE = "select DBS.name database_name, TBLS.tbl_name table_name, part.part_name partition_name, partition_key.pkey_name partition_key from PARTITIONS part join PART_RANGES partition_range on part.part_id = partition_range.part_id join TBLS on part.tbl_id = TBLS.tbl_id join DBS on TBLS.db_id = DBS.db_id join PARTITION_KEYS partition_key on partition_key.tbl_id = TBLS.tbl_id;";
	
	private final String hdfsName;
	private final Map<String, OrcTableBean> orcTables;
	private final Connection conn;
	
	public OrcTableCheckOnDetail(final String hdfsName,
			final Map<String, OrcTableBean> tables,
			final JSONObject connectionInfo) {
		this.hdfsName = hdfsName;
		this.orcTables = tables;
		this.conn = JDBCConnected.getConnection(connectionInfo.getString("url"), 
				connectionInfo.getString("username"), 
				connectionInfo.getString("password"));
	}

	@Override
	public void addTableCheck() throws Exception {
		try {
			getAllBucketInfo();
			getAllPartitionInfo(QUERY_PARTITIONS, false);
			getAllPartitionInfo(QUERY_PARTITIONS_RANGE, true);
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			this.conn.close();
		}
		for (Entry<String, OrcTableBean> entry : orcTables.entrySet()) {
			OrcTableBean table = entry.getValue();
			MyThreadPool.addNewThread(new OrcTableCheckRunnable(hdfsName, table));
		}
	}
	
	private void getAllBucketInfo() {
		PreparedStatement pstat = null;
		ResultSet rs = null;
		try {
			pstat = conn.prepareStatement(QUERY_BUCKETS);
			rs = pstat.executeQuery();
			while (rs.next()) {
				String tablename = rs.getString("table_name");
				String database = rs.getString("database_name");
				OrcTableBean table = orcTables.get(String.format("%s.%s", database, tablename));
				if (table == null) {
					log.error(String.format("no find table [%s.%s]", database, tablename));
					continue;
				}
				table.setBucketColumn(rs.getObject("bucket_column"));
				table.setBucketNumber(rs.getObject("bucket_number"));
			}
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			close(pstat, rs);
		}
	}
	
	private void getAllPartitionInfo(final String sql, final Boolean isRange) {
		PreparedStatement pstat = null;
		ResultSet rs = null;
		try {
			pstat = conn.prepareStatement(sql);
			rs = pstat.executeQuery();
			while (rs.next()) {
				String database = rs.getString("database_name");
				String tablename = rs.getString("table_name");
				String topic = String.format("%s.%s", database, tablename);
				OrcTableBean table = orcTables.get(topic);
				if (table == null) {
					log.error(String.format("no find table [%s]", topic));
					continue;
				}
				table.setPartitionRange(isRange);
				table.setPartitionKey(rs.getObject("partition_key"));
				table.addPartition(rs.getObject("partition_name"));
			}
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			close(pstat, rs);
		}
	}
	
	private void close(final PreparedStatement pstat, final ResultSet rs) {
		if (rs != null) {
			try {
				rs.close();
			} catch (Exception e) {}
		}
		if (pstat != null) {
			try {
				pstat.close();
			} catch (Exception e) {}
		}
	}
}
