package io.transwarp.table.orc.servlet;

import java.io.FileNotFoundException;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;

import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.transwarp.common.GlobalArgs;
import io.transwarp.common.bean.table.OrcTableBean;
import io.transwarp.common.util.AbstractRunnable;
import io.transwarp.common.util.HdfsConnUtils;
import io.transwarp.inspection.CheckItemEnum;

public class OrcTableCheckRunnable extends AbstractRunnable {
	
	private final static Logger log = LoggerFactory.getLogger(OrcTableCheckRunnable.class);
	private final OrcTableBean table;
	private final HdfsConnUtils hdfsPool;
	private FileSystem fs;
	
	public OrcTableCheckRunnable(final String hdfsName,
			final OrcTableBean table) {
		this.hdfsPool = GlobalArgs.hdfsPools.get(hdfsName);
		this.table = table;
	}
	
	@Override
	public void startCheck() throws Exception {
		fs = hdfsPool.getHdfsConn();
		try {
			List<String> partitions = table.getPartitionNames();
			String location = HdfsConnUtils.getHdfsPath(table.getLocation());
			if (partitions.size() == 0) {
				partitions.add("");
			}
			for (String partition : partitions) {
				checkPartitionSpace(location, partition);
			}		
		} catch (FileNotFoundException e) {
			log.warn("no find table [{}] location path on hdfs [{}]", String.format("%s.%s", table.getDatabase(), table.getTablename()), table.getLocation());
		}
	}
	
	@Override
	public void dealWithError(Exception e) {
		String msg = String.format("check orc table [%s.%s] is faild | %s", table.getDatabase(), table.getTablename(), e.getMessage());
		GlobalArgs.ERROR_INFO.add(msg);
		log.error(msg);
		e.printStackTrace();
	}
	
	@Override
	public void closeThread() {
		hdfsPool.closeHdfsConn(fs);
		GlobalArgs.timeStatisticses.get(CheckItemEnum.TABLE.name()).setEndTime();
	}
	
	private void checkPartitionSpace(final String tableLocation, final String partition) throws Exception {
		int deltaMenuNumber = 0;
		Queue<String> queue = new LinkedList<String>();
		queue.offer(tableLocation + "/" + partition);
		while (!queue.isEmpty()) {
			String checkPath = queue.poll();
			FileStatus[] files = fs.listStatus(new Path(checkPath));
			for (FileStatus file : files) {
				if (file.isDirectory()) {
					String dirName = file.getPath().getName();
					if (dirName.startsWith("delta")) {
						deltaMenuNumber += 1;
					}
					queue.offer(file.getPath().toString());
				} else {
					Long fileSize = file.getLen();
					String filename = file.getPath().getName();
					table.addDataFile(partition, filename, fileSize);
				}
			}
		}
		table.addDeltaMenu(partition, deltaMenuNumber);		
	}
}
