package io.transwarp.table.servlet;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.transwarp.common.GlobalArgs;
import io.transwarp.common.bean.TableInfo;
import io.transwarp.common.bean.service.ServiceBean;
import io.transwarp.common.bean.table.*;
import io.transwarp.common.util.AbstractRunnable;
import io.transwarp.common.util.JDBCConnected;
import io.transwarp.service.servlet.ServiceType;
import net.sf.json.JSONObject;

public class AllTableCheckRunnable extends AbstractRunnable {
	
	private final Logger log = LoggerFactory.getLogger(AllTableCheckRunnable.class);
	/** mysql数据库查询元数据 */
	public static final String QUERY_TABLE_MYSQL = "SELECT tbls.TBL_NAME AS table_name, DBS.NAME AS database_name, tbls.TBL_TYPE AS table_type, tbls.OWNER AS owner_name, temp_hbases_v.hbase_name, temp_es_v.es_table, (CASE WHEN isnull(temp_transactional_v.param_value) THEN 'false' ELSE temp_transactional_v.param_value END) AS transactional, (CASE  WHEN (SDS.INPUT_FORMAT = 'org.apache.hadoop.mapred.TextInputFormat') THEN 'text' WHEN (SDS.INPUT_FORMAT = 'org.apache.hadoop.mapred.SequenceFileInputFormat') THEN 'sequence' WHEN (SDS.INPUT_FORMAT = 'org.apache.hadoop.hive.ql.io.orc.OrcInputFormat') THEN 'orc' WHEN (SDS.INPUT_FORMAT = 'io.transwarp.inceptor.memstore2.MemoryTableInputFormat') THEN 'memory' WHEN (SDS.INPUT_FORMAT = 'io.transwarp.hyperdrive.HyperdriveInputFormat') THEN 'hyperdrive' WHEN (SDS.INPUT_FORMAT = 'org.apache.hadoop.hive.ql.io.parquet.MapredParquetInputFormat') THEN 'parquet' WHEN (SDS.INPUT_FORMAT = 'org.apache.hadoop.hive.ql.io.RCFileInputFormat') THEN 'rcfile' WHEN (SDS.INPUT_FORMAT = 'io.transwarp.hyperbase.HyperbaseInputFormat') THEN 'hyperbase' WHEN (SDS.INPUT_FORMAT = 'org.apache.hadoop.hive.hbase.HiveHBaseTableInputFormat') THEN 'hbase' WHEN (SDS.INPUT_FORMAT = 'io.transwarp.esdrive.ElasticSearchInputFormat') THEN 'es' ELSE SDS.INPUT_FORMAT END) AS table_format, SDS.LOCATION AS table_location FROM((((SELECT TBLS.DB_ID AS DB_ID, TBLS.SD_ID AS SD_ID, TBLS.TBL_ID AS TBL_ID, TBLS.OWNER AS OWNER, TBLS.TBL_NAME AS TBL_NAME, TBLS.TBL_TYPE AS TBL_TYPE FROM TBLS WHERE TBLS.TBL_TYPE <> 'VIRTUAL_VIEW') tbls JOIN DBS ON tbls.DB_ID = DBS.DB_ID ) JOIN SDS ON tbls.SD_ID = SDS.SD_ID )  LEFT JOIN(SELECT TABLE_PARAMS.TBL_ID AS tbl_id, TABLE_PARAMS.PARAM_VALUE AS param_value FROM TABLE_PARAMS  WHERE TABLE_PARAMS.PARAM_KEY = 'transactional') temp_transactional_v  ON tbls.TBL_ID = temp_transactional_v.tbl_id  LEFT JOIN(SELECT TABLE_PARAMS.TBL_ID AS tbl_id, TABLE_PARAMS.PARAM_VALUE AS hbase_name FROM TABLE_PARAMS WHERE TABLE_PARAMS.PARAM_KEY = 'hbase.table.name') temp_hbases_v ON tbls.TBL_ID = temp_hbases_v.tbl_id  LEFT JOIN(SELECT TABLE_PARAMS.TBL_ID AS tbl_id, TABLE_PARAMS.PARAM_VALUE AS es_table FROM TABLE_PARAMS WHERE TABLE_PARAMS.PARAM_KEY = 'elasticsearch.tablename') temp_es_v ON tbls.TBL_ID = temp_es_v.tbl_id);";
	
	private final ServiceBean inceptorService;
	private final ServiceBean hdfsService;
	private final ServiceBean hyperbaseService;
	private final ServiceBean searchService;
	private final ServiceBean zookeeperService;
	private final JSONObject connectionInfo;
	private final TableInfo tableInfo;
	
	private final Map<String, OrcTableBean> orcTables;
	private final Map<String, HBaseTableBean> hbaseTables;
	private final Map<String, HolodeskTableBean> holodeskTables;
	private final Map<String, EsTableBean> esTables;
	private final Map<String, TextTableBean> textTables;
	private final Map<String, TextTableBean> otherTables;

	private Connection conn;
	
	public AllTableCheckRunnable(final ServiceBean inceptorService,
			final Map<ServiceType, ServiceBean> dependencyServices,
			final JSONObject connectionInfo,
			final TableInfo tableInfo) {
		this.inceptorService = inceptorService;
		this.hdfsService = dependencyServices.get(ServiceType.HDFS);
		this.hyperbaseService = dependencyServices.get(ServiceType.HYPERBASE);
		this.searchService = dependencyServices.get(ServiceType.SEARCH);
		this.zookeeperService = dependencyServices.get(ServiceType.ZOOKEEPER);
		this.connectionInfo = connectionInfo;
		this.tableInfo = tableInfo;
		
		this.orcTables = new ConcurrentHashMap<String, OrcTableBean>();
		this.hbaseTables = new ConcurrentHashMap<String, HBaseTableBean>();
		this.holodeskTables = new ConcurrentHashMap<String, HolodeskTableBean>();
		this.esTables = new ConcurrentHashMap<String, EsTableBean>();
		this.textTables = new ConcurrentHashMap<String, TextTableBean>();
		this.otherTables = new ConcurrentHashMap<String, TextTableBean>();
	}
	
	@Override
	public void startCheck() throws Exception {
		StringBuffer checkInfo = new StringBuffer();
		checkInfo.append("begin check table -------------------").append("\n")
			.append("inceptor service is : [").append(inceptorService.getName()).append("]\n")
			.append("hdfs service is : [").append(hdfsService == null ? "" : hdfsService.getName()).append("]\n")
			.append("hyperbase service is : [").append(hyperbaseService == null ? "" : hyperbaseService.getName()).append("]\n")
			.append("search service is : [").append(searchService == null ? "" : searchService.getName()).append("]\n")
			.append("zookeeper service is : [").append(zookeeperService == null ? "" : zookeeperService.getName()).append("]\n")
			.append("mysql url is [").append(connectionInfo.getString("url")).append("]\n");
		log.info(checkInfo.toString());
		conn = JDBCConnected.getConnection(this.connectionInfo.getString("url"),
				this.connectionInfo.getString("username"),
				this.connectionInfo.getString("password"));
		if (conn != null) {
			queryTableInfoFromMysql(conn);
			addResultToTableInfo();
		}
		checkTableOnDetail();
	}
	
	@Override
	public void dealWithError(Exception e) {
		String errorMessage = String.format("check table info of service [%s] is faild | %s", inceptorService.getName(), e.getMessage());
		GlobalArgs.ERROR_INFO.add(errorMessage);
		log.error(errorMessage);
		e.printStackTrace();		
	}
	
	@Override
	public void closeThread() {
		try {
			if (conn != null) {
				conn.close();
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	private void queryTableInfoFromMysql(Connection conn) {
		PreparedStatement pstat = null;
		ResultSet rs = null;
		try {
			pstat = conn.prepareStatement(QUERY_TABLE_MYSQL);
			rs = pstat.executeQuery();
			while (rs.next()) {
				String database = rs.getString("database_name");
				String tablename = rs.getString("table_name");
				String format = rs.getString("table_format");
				String transactional = rs.getString("transactional");
				String hbase_name = rs.getString("hbase_name");
				String es_name = rs.getString("es_table");
				AbstractTable table = getTable(database, tablename, format, transactional, hbase_name, es_name);
				table.setDatabase(database);
				table.setTablename(tablename);
				table.setFormat(format);
				table.setLocation(rs.getObject("table_location"));
				table.setTableType(rs.getObject("table_type"));
				table.setOwner(rs.getObject("owner_name"));
			}
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			close(pstat, rs);
		}
	}
	
	private void addResultToTableInfo() {
		if (inceptorService != null) {
			tableInfo.TABLE_ORC.addOrcTables(inceptorService.getName(), orcTables);
			tableInfo.TABLE_HOLODESK.addHolodeskTables(inceptorService.getName(), holodeskTables);
			tableInfo.TABLE_TEXT.addTextTables(inceptorService.getName(), textTables);
			tableInfo.addOtherTables(inceptorService.getName(), otherTables);
		}
		if (hyperbaseService != null) {
			tableInfo.TABLE_HBASE.addHbaseTables(hyperbaseService.getName(), hbaseTables);
		}
		if (searchService != null) {
			tableInfo.TABLE_ES.addEsTables(searchService.getName(), esTables);
		}
	}
	
	private AbstractTable getTable(final String database,
			final String tablename,
			final String format,
			final String transactional,
			final String hbase_name,
			final String es_name) {
		String name = String.format("%s.%s", database, tablename);
		AbstractTable table = null;
		if (format.equals("orc") || format.equals("rcfile") || format.equals("parquet")) {
			table = new OrcTableBean();
			((OrcTableBean)table).setTransactional(transactional);
			orcTables.put(name, (OrcTableBean)table);
		} else if (format.equals("memory")) {
			table = new HolodeskTableBean();
			holodeskTables.put(name, (HolodeskTableBean)table);
		} else if (format.equals("hbase") || format.equals("hyperdrive") || format.equals("org.apache.hadoop.hive.hbase.transaction.HBaseTransactionInputFormat")) {
			table = new HBaseTableBean();
			((HBaseTableBean)table).setHBaseTablename(hbase_name);			
			hbaseTables.put(name, (HBaseTableBean)table);
		} else if (format.equals("es")) {
			table = new EsTableBean();
			((EsTableBean)table).setEsTablename(es_name);
			esTables.put(name, (EsTableBean)table);
		} else if (format.equals("text")) {
			table = new TextTableBean();
			textTables.put(name, (TextTableBean)table);
		} else {
			table = new TextTableBean();
			otherTables.put(name, (TextTableBean)table);
		}
		return table;
	}
	
	private void checkTableOnDetail() {
		AbstractCheckOnDetail.Build build = AbstractCheckOnDetail.custom()
				.setInceptor(inceptorService)
				.setHdfs(hdfsService)
				.setHyperbase(hyperbaseService)
				.setSearch(searchService)
				.setZookeeper(zookeeperService)
				.setTextTables(textTables)
				.setOtherTables(otherTables)
				.setOrcTables(orcTables)
				.setHBaseTables(hbaseTables)
				.setHolodeskTables(holodeskTables)
				.setEsTables(esTables)
				.setConnectedInfo(connectionInfo)
				.setProperty(tableInfo);
		AbstractCheckOnDetail check = null;
		for (TableTypeEnum type : TableTypeEnum.values()) {
			try {
				check = build.build(type);
				if (check == null) {
					continue;
				}
				check.addTableCheck();
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	private void close(final PreparedStatement pstat, final ResultSet rs) {
		if (rs != null) {
			try {
				rs.close();
			} catch (Exception e) {}
		}
		if (pstat != null) {
			try {
				pstat.close();
			} catch (Exception e) {}
		}
	}
	
}
