package io.transwarp.table.servlet;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.sf.json.JSONObject;
import io.transwarp.common.GlobalArgs;
import io.transwarp.common.bean.TableInfo;
import io.transwarp.common.bean.service.ConfigBean;
import io.transwarp.common.bean.service.RoleBean;
import io.transwarp.common.bean.service.ServiceBean;
import io.transwarp.common.bean.table.EsTableBean;
import io.transwarp.common.bean.table.HBaseTableBean;
import io.transwarp.common.util.MyThreadPool;
import io.transwarp.inspection.CheckInterface;
import io.transwarp.service.servlet.ServiceType;
import io.transwarp.table.es.servlet.EsTableCheckOnDetail;
import io.transwarp.table.hbase.servlet.HBaseTableCheckOnDetail;

public class TableCheck implements CheckInterface{
	
	private final Logger log = LoggerFactory.getLogger(TableCheck.class);
	private final TableInfo tableInfo;
	private final Map<String, ServiceBean> services;
	private final Set<String> hasCheckHyperbase;
	private final Set<String> hasCheckSearch;
	
	public TableCheck(final TableInfo tableInfo,
			final Map<String, ServiceBean> services) {
		this.tableInfo = tableInfo;
		this.services = services;
		this.hasCheckHyperbase = new HashSet<String>();
		this.hasCheckSearch = new HashSet<String>();
	}

	@Override
	public void check() throws Exception {
		getTableFromInceptor();
		if (GlobalArgs.checkSelect.isHbaseCheck()) {
			getTablesOfHyperbase();
		}
		if (GlobalArgs.checkSelect.isEsCheck() || GlobalArgs.checkSelect.isServiceCheckEs()) {
			getTablesOfSearch();
		}
	}
	
	private void getTableFromInceptor() {
		for (Entry<String, ServiceBean> entry : services.entrySet()) {
			ServiceBean inceptorService = entry.getValue();
			String metaStoreIP = getMetaStoreIP(inceptorService);
			if (metaStoreIP == null) {
				continue;
			}
			Map<ServiceType, ServiceBean> dependencyServices = getDependencyService(inceptorService.getDependencies());
			ServiceBean hdfsService = dependencyServices.get(ServiceType.HDFS);
			if (!GlobalArgs.hdfsPools.get(hdfsService.getName()).isBuildPoolSuccess()) {
				log.error("hdfs connection pool did not open of service : " + hdfsService.getName());
				continue;
			}

			ServiceBean hyperbaseService = dependencyServices.get(ServiceType.HYPERBASE);
			ServiceBean searchService = dependencyServices.get(ServiceType.SEARCH);
			if (hyperbaseService != null) {
				this.hasCheckHyperbase.add(hyperbaseService.getName());
			}
			if (searchService != null) {
				this.hasCheckSearch.add(searchService.getName());
			}
			
			try {
				ConfigBean config = null;
				int count = 0;
				while (config == null) {
					config = inceptorService.getConfig();
					if (config == null) {
						Thread.sleep(2000);
						count += 1;
					}
					if (count > 150) {
						break;
					}
				}
				if (config == null) {
					log.error("check config of service [{}] over time", inceptorService.getName());
					continue;
				}
				JSONObject connectionInfo = getConnectionInfo(inceptorService.getName(), metaStoreIP, config);
				MyThreadPool.addNewThread(new AllTableCheckRunnable(inceptorService, dependencyServices, connectionInfo, tableInfo));
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	private void getTablesOfHyperbase() {
		for (Entry<String, ServiceBean> entry : services.entrySet()) {
			String servicename = entry.getKey();
			if (this.hasCheckHyperbase.contains(servicename)) {
				continue;
			}
			ServiceBean service = entry.getValue();
			if (service.getType().matches("\\S*HYPERBASE\\S*")) {
				this.hasCheckHyperbase.add(servicename);
				Map<ServiceType, ServiceBean> dependencyServices = getDependencyService(service.getDependencies());
				ServiceBean hdfsService = dependencyServices.get(ServiceType.HDFS);
				if (!GlobalArgs.hdfsPools.get(hdfsService.getName()).isBuildPoolSuccess()) {
					log.error("hdfs connection pool did not open of service : " + hdfsService.getName());
					continue;
				}
				Map<String, HBaseTableBean> hbaseTables = new ConcurrentHashMap<String, HBaseTableBean>();
				tableInfo.TABLE_HBASE.addHbaseTables(servicename, hbaseTables);
				HBaseTableCheckOnDetail hbaseCheck = new HBaseTableCheckOnDetail(hdfsService.getName(), service, hbaseTables, tableInfo);
				try {
					hbaseCheck.addTableCheck();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
	}
	
	private void getTablesOfSearch() {
		for (Entry<String, ServiceBean> entry : services.entrySet()) {
			String servicename = entry.getKey();
			if (this.hasCheckSearch.contains(servicename)) {
				continue;
			}
			ServiceBean service = entry.getValue();
			if (service.getType().matches("\\S*SEARCH\\S*")) {
				this.hasCheckSearch.add(servicename);
				Map<String, EsTableBean> esTables = new ConcurrentHashMap<String, EsTableBean>();
				tableInfo.TABLE_ES.addEsTables(servicename, esTables);
				EsTableCheckOnDetail esCheck = new EsTableCheckOnDetail(service, esTables, tableInfo);
				try {
					esCheck.addTableCheck();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
	}
	
	private String getMetaStoreIP(ServiceBean service) {
		String serviceType = service.getType();
		if (!serviceType.matches("\\S*INCEPTOR\\S*")) {
			return null;
		}
		List<RoleBean> roles = service.getRoles();
		for (RoleBean role : roles) {
			String roleType = role.getRoleType();
			if (roleType.matches("\\S*METASTORE\\S*")) {
				return role.getNode().getIpAddress();
			}
		}
		return null;
	}
	
	private JSONObject getConnectionInfo(final String servicename, 
			final String metaStoreIP, 
			final ConfigBean config) throws Exception {
		Map<String, String> configFile = config.getConfigFile(metaStoreIP, "hive-site.xml");
		if (configFile == null) {
			log.error("no find inceptor config file : hive-site.xml");
			return null;
		}
		String url = configFile.get("javax.jdo.option.ConnectionURL");
		String username = configFile.get("javax.jdo.option.ConnectionUserName");
		String password = configFile.get("javax.jdo.option.ConnectionPassword");
		if (url == null || username == null || password == null) {
			throw new Exception("no find connection info of service : " + servicename);
		}
		JSONObject connectionInfo = new JSONObject();
		connectionInfo.put("url", url.split("\\?")[0] + "?characterEncoding=utf-8&tcpRcvBuf=1024000");
		connectionInfo.put("username", username);
		connectionInfo.put("password", password);
		return connectionInfo;
	}
	
	private Map<ServiceType, ServiceBean> getDependencyService(final String dependency) {
		String[] ids = dependency.substring(1, dependency.length() - 1).split(",");
		Map<ServiceType, ServiceBean> dependencies = new HashMap<ServiceType, ServiceBean>();
		for (Entry<String, ServiceBean> entry : services.entrySet()) {
			ServiceBean service = entry.getValue();
			String id = service.getId();
			String serviceType = service.getType();
			for (String dependencyId : ids) {
				if (dependencyId.equals(id)) {
					dependencies.put(getServiceType(serviceType), service);
				}
			}
		}
		return dependencies;
	}
	
	private ServiceType getServiceType(String type) {
		if (type.matches("\\S*HYPERBASE\\S*")) {
			return ServiceType.HYPERBASE;
		} else if (type.matches("\\S*ZOOKEEPER\\S*")) {
			return ServiceType.ZOOKEEPER;
		} else if (type.matches("\\S*HDFS\\S*")) {
			return ServiceType.HDFS;
		} else if (type.matches("\\S*SEARCH\\S*")) {
			return ServiceType.SEARCH;
		} else if (type.matches("\\S*GUARDIAN\\S*")) {
			return ServiceType.GUARDIAN;
		} else if (type.matches("\\S*YARN\\S*")) {
			return ServiceType.YARN;
		} else if (type.matches("\\S*TXSQL\\S*")) {
			return ServiceType.TXSQL;
		} else if (type.matches("\\S*SHIVA\\S*")) {
			return ServiceType.SHIVA;
		}
		return ServiceType.OTHER;
	}

}
